<?php

namespace ZiziCache;

/**
 * Handles the WordPress admin bar integration for ZiziCache.
 * 
 * This class is responsible for adding cache management options to the WordPress admin bar,
 * including cache purging, preloading, and quick access to settings.
 */
class AdminBar
{
  /**
   * Initializes the admin bar functionality.
   * 
   * Sets up the necessary hooks for adding menu items to the admin bar
   * and enqueuing required scripts and styles.
   * 
   * @return void
   */
  /**
   * Initializes the admin bar integration for ZiziCache.
   *
   * Adds menu items to the admin bar and enqueues required scripts and styles for both frontend and backend.
   *
   * @return void
   */
  public static function init()
  {
    // Add items to the admin bar
    add_action('admin_bar_menu', [__CLASS__, 'add_menu_items'], 10000);

    // Enqueue scripts and localize settings
    add_action('wp_enqueue_scripts', [__CLASS__, 'enqueue_admin_bar_scripts']); // Frontend hook
    add_action('admin_enqueue_scripts', [__CLASS__, 'enqueue_admin_bar_scripts']); // Backend hook
  }

  /**
   * Enqueues scripts and styles for the admin bar.
   * 
   * Loads the necessary JavaScript and CSS files for the admin bar functionality,
   * but only if the admin bar is showing and the current user has the required permissions.
   * 
   * @return void
   */
  /**
   * Enqueues scripts and styles for the admin bar.
   *
   * Loads the necessary JavaScript and CSS files for the admin bar functionality,
   * but only if the admin bar is showing and the current user has the required permissions.
   *
   * @return void
   */
  /**
   * Cache for Redis detection to prevent repeated filesystem calls.
   * 
   * @var bool|null Cached result of Redis detection (true = active, false = inactive, null = not checked yet)
   */
  private static $redis_active = null;

  public static function enqueue_admin_bar_scripts()
  {
    // Only enqueue if admin bar is showing and user has required permissions
    if (!is_admin_bar_showing() || !Authority::is_allowed()) {
      return;
    }

    // Skip enqueuing on WooCommerce admin pages to prevent JavaScript conflicts
    if (defined('ZIZI_CACHE_ADMIN_BYPASS') && ZIZI_CACHE_ADMIN_BYPASS) {
      return;
    }

    // Use WooCommerce compatibility layer for additional detection
    if (class_exists('\\ZiziCache\\WooCommerceCompatibility') && WooCommerceCompatibility::is_woocommerce_admin_page()) {
      return;
    }

    // Additional WooCommerce detection if bypass wasn't set during bootstrap
    if (isset($_GET['page']) && $_GET['page'] === 'wc-admin') {
      return;
    }
    // Log only in debug mode
    if (defined('WP_DEBUG') && WP_DEBUG) {
      \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache AdminBar: enqueue_admin_bar_scripts called.');
    }
    // Define script handle and asset path
    $script_handle = 'zizi-cache-admin-bar';
    $script_url = ZIZI_CACHE_PLUGIN_URL . 'assets/js/admin-bar.js';
    $script_version = defined('ZIZI_CACHE_VERSION') ? ZIZI_CACHE_VERSION : false; // Add version for cache busting

    // Enqueue script, assuming dependency on 'admin-bar'.
    // If there are other dependencies (e.g. jQuery), add them to the array.
    \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache AdminBar: Script URL: ' . $script_url); // DEBUG LOG
    \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache AdminBar: Script Version: ' . print_r($script_version, true)); // DEBUG LOG
    wp_enqueue_script($script_handle, $script_url, ['admin-bar'], $script_version, true); // Load in footer

    // Enqueue CSS file
    $css_url = ZIZI_CACHE_PLUGIN_URL . 'assets/css/admin-bar.css';
    wp_enqueue_style('zizi-cache-admin-bar', $css_url, [], $script_version);

    // Prepare settings for localization
    $plugin_dir_name = basename(dirname(ZIZI_CACHE_FILE)); // Get plugin directory name
    $settings = [
      'restUrl' => rtrim(get_rest_url(null, $plugin_dir_name), '/'), // Use dynamic namespace
      'nonce'   => wp_create_nonce('wp_rest'), // Create nonce for REST API
    ];

    // Localize script with settings
    \ZiziCache\CacheSys::writeLog('[INFO] ZiziCache AdminBar: Localizing settings: ' . print_r($settings, true)); // DEBUG LOG
    wp_localize_script($script_handle, 'ziziCacheSettings', $settings);
  }

  /**
   * Adds menu items to the WordPress admin bar.
   * 
   * Creates a menu structure in the admin bar with various cache management options
   * including cache purging, preloading, and access to settings.
   * 
   * @param \WP_Admin_Bar $admin_bar The WordPress admin bar object.
   * @return void
   */
  /**
   * Adds menu items to the WordPress admin bar.
   *
   * Creates a menu structure in the admin bar with various cache management options
   * including cache purging, preloading, and access to settings. All menu item labels and warnings
   * are provided in English for internationalization.
   *
   * @param \WP_Admin_Bar $admin_bar The WordPress admin bar object.
   * @return void
   */
  public static function add_menu_items($admin_bar)
  {
    if (!Authority::is_allowed()) {
      return;
    }

    $admin_bar->add_menu([
      'id' => 'zizi-cache',
      'title' => 'ZiziCache',
      'href' => admin_url('admin.php?page=zizi-cache'),
      'meta' => [
        'class' => 'ab-item',
      ],
    ]);

    if (!is_admin()) {
      $admin_bar->add_menu([
        'id'     => 'purge-current-page',
        'parent' => 'zizi-cache',
        'title'  => __('Purge cache for this page', 'zizi-cache'),
        'href'   => '#',
        'meta'   => [
          'class'        => 'ab-item',
          'data-action'  => 'purge-current-page',
          'onclick'      => 'return false;',
        ],
      ]);
    }

    // --- Warning for long queue of expired URLs ---
    $sqlite_path = ZIZI_CACHE_CACHE_DIR . 'preload.sqlite';
    if (file_exists($sqlite_path)) {
      try {
        $db = new \PDO('sqlite:' . $sqlite_path);
        $count_pending = $db->query('SELECT COUNT(*) FROM zizi_expired_cache WHERE status = "pending"')->fetchColumn();
        $count_failed = $db->query('SELECT COUNT(*) FROM zizi_expired_cache WHERE status = "failed"')->fetchColumn();
        if ($count_pending > 100) {
          $admin_bar->add_menu([
            'id'     => 'zizi-cache-preload-warning',
            'parent' => 'zizi-cache',
            'title'  => sprintf(__('Warning: There are %d expired URLs in the preload queue!', 'zizi-cache'), $count_pending),
            'href'   => '#',
            'meta'   => [
              'class' => 'ab-item zizi-cache-warning',
            ],
          ]);
        }
        if ($count_failed > 0) {
          $admin_bar->add_menu([
            'id'     => 'zizi-cache-preload-failed',
            'parent' => 'zizi-cache',
            'title'  => sprintf(__('Error: %d URLs could not be restored!', 'zizi-cache'), $count_failed),
            'href'   => '#',
            'meta'   => [
              'class' => 'ab-item zizi-cache-failed',
            ],
          ]);
        }
      } catch (\Exception $e) {}
    }

    $admin_bar->add_menu([
      'id'     => 'purge-pages',
      'parent' => 'zizi-cache',
      'title'  => __('Purge page cache', 'zizi-cache'),
      'href'   => '#',
      'meta'   => [
        'class'        => 'ab-item',
        'data-action'  => 'purge-pages',
        'onclick'      => 'return false;',
      ],
    ]);

    $admin_bar->add_menu([
      'id'     => 'preload-cache',
      'parent' => 'zizi-cache',
      'title'  => __('Preload cache', 'zizi-cache'),
      'href'   => '#',
      'meta'   => [
        'class'        => 'ab-item',
        'data-action'  => 'preload-cache',
        'onclick'      => 'return false;',
      ],
    ]);

    $admin_bar->add_menu([
      'id'     => 'purge-pages-and-preload',
      'parent' => 'zizi-cache',
      'title'  => __('Purge page cache and preload', 'zizi-cache'),
      'href'   => '#',
      'meta'   => [
        'class'        => 'ab-item',
        'data-action'  => 'purge-pages-and-preload',
        'onclick'      => 'return false;',
      ],
    ]);

    $admin_bar->add_menu([
      'id'     => 'purge-everything-and-preload',
      'parent' => 'zizi-cache',
      'title'  => __('Purge everything and preload', 'zizi-cache'),
      'href'   => '#',
      'meta'   => [
        'class'        => 'ab-item',
        'data-action'  => 'purge-everything-and-preload',
        'onclick'      => 'return false;',
      ],
    ]);

    $admin_bar->add_menu([
      'id'     => 'purge-critical-css',
      'parent' => 'zizi-cache',
      'title'  => __('Clear Critical CSS Cache', 'zizi-cache'),
      'href'   => '#',
      'meta'   => [
        'class'        => 'ab-item',
        'data-action'  => 'purge-critical-css',
        'onclick'      => 'return false;',
      ],
    ]);

    // --- Divider before Object Cache/OPCache ---
    $admin_bar->add_menu([
      'id'     => 'zizi-cache-divider-redis-opcache',
      'parent' => 'zizi-cache',
      'title'  => '',
      'href'   => false,
      'meta'   => [
        'class' => 'zizi-cache-divider',
        'html'  => '',
        'group' => true,
      ],
    ]);

    // Add item for purging Redis Object Cache if active (runtime detection)
    if (self::is_redis_active()) {
      $admin_bar->add_menu([
        'id'     => 'purge-object-cache',
        'parent' => 'zizi-cache',
        'title'  => __('Purge Object Cache Redis', 'zizi-cache'),
        'href'   => '#',
        'meta'   => [
          'class'        => 'ab-item zizi-cache-redis',
          'data-action'  => 'redis-flush',
          'onclick'      => 'return false;',
        ],
      ]);
    }

    $admin_bar->add_menu([
      'id'     => 'purge-opcache',
      'parent' => 'zizi-cache',
      'title'  => __('Purge OPCache', 'zizi-cache'),
      'href'   => '#',
      'meta'   => [
        'class'        => 'ab-item zizi-cache-opcache',
        'data-action'  => 'opcache/flush',
        'onclick'      => 'return false;',
      ],
    ]);

    // --- Divider before Settings ---
    $admin_bar->add_menu([
      'id'     => 'zizi-cache-divider-settings',
      'parent' => 'zizi-cache',
      'title'  => '',
      'href'   => false,
      'meta'   => [
        'class' => 'zizi-cache-divider',
        'html'  => '',
        'group' => true,
      ],
    ]);

    // Last item: Settings
    $admin_bar->add_menu([
      'id'     => 'zizi-cache-settings',
      'parent' => 'zizi-cache',
      'title'  => __('Settings', 'zizi-cache'),
      'href'   => admin_url('admin.php?page=zizi-cache'),
      'meta'   => [
        'class' => 'ab-item zizi-cache-settings',
      ],
    ]);
  }

  /**
   * Detects if Redis Object Cache is active.
   * The result is cached in a static variable for better performance.
   *
   * This method checks for Redis in the following order:
   * 1. Checks if object-cache.php exists and contains 'redis' string
   * 2. Checks if Redis_Object_Cache class exists
   * 3. Checks for WP Redis configuration
   *
   * @return bool True if Redis is active, false otherwise
   */
  private static function is_redis_active() {
    // Return cached result if available
    if (self::$redis_active !== null) {
      return self::$redis_active;
    }

    // Perform Redis detection (only once per request)
    $object_cache_path = WP_CONTENT_DIR . '/object-cache.php';
    self::$redis_active = (
        (file_exists($object_cache_path) && stripos(@file_get_contents($object_cache_path), 'redis') !== false) ||
        class_exists('Redis_Object_Cache') ||
        (function_exists('wp_cache_flush') && defined('WP_REDIS_VERSION'))
    );

    return self::$redis_active;
  }
}
