<?php
namespace ZiziCache;

/**
 * Handles cleanup of Action Scheduler tables in WordPress.
 * 
 * This class provides functionality to clean up and optimize Action Scheduler tables,
 * schedule automatic cleanups, and provide status information about the tables.
 */
class ActionSchedulerCleanup
{
    /**
     * Initializes the class by setting up necessary hooks.
     * 
     * This method registers the necessary WordPress hooks for scheduling and performing
     * Action Scheduler cleanup tasks.
     * 
     * @return void
     */
    public static function init()
    {
        // Add dynamic cron schedule for AS cleanup
        add_filter('cron_schedules', [__CLASS__, 'add_as_cleanup_schedule']);
        // Schedule or clear AS cleanup event on init
        add_action('init', [__CLASS__, 'setup_scheduled_cleanup']);
        // Hook AS cleanup event
        add_action('zizi_cache_actionscheduler_cleanup', [__CLASS__, 'cleanup']);
        // Also run cleanup after DB cleanup for backward compatibility
        add_action('zizi_cache_database_after_options', [__CLASS__, 'maybe_run_cleanup']);
    }

    /**
     * Adds a dynamic cron schedule for Action Scheduler cleanup based on log retention settings.
     * 
     * @param array $schedules Existing WordPress cron schedules.
     * @return array Modified cron schedules with the new AS cleanup interval.
     */
    public static function add_as_cleanup_schedule($schedules)
    {
        $config = SysConfig::$config;
        if (!empty($config['db_actionscheduler_clean'])) {
            $days = max(1, intval($config['db_actionscheduler_logs_days']));
            $schedules['as_cleanup_interval'] = [
                'interval' => $days * DAY_IN_SECONDS,
                'display'  => sprintf('AS Cleanup every %d day(s)', $days),
            ];
        }
        return $schedules;
    }

    /**
     * Sets up or clears the scheduled cleanup event based on configuration.
     * 
     * This method checks the configuration and either schedules or unschedules
     * the Action Scheduler cleanup event accordingly.
     * 
     * @return void
     */
    public static function setup_scheduled_cleanup()
    {
        $config = SysConfig::$config;
        $hook   = 'zizi_cache_actionscheduler_cleanup';
        if (empty($config['db_actionscheduler_clean'])) {
            wp_clear_scheduled_hook($hook);
            return;
        }
        if (!wp_next_scheduled($hook) || wp_get_schedule($hook) !== 'as_cleanup_interval') {
            wp_clear_scheduled_hook($hook);
            wp_schedule_event(time(), 'as_cleanup_interval', $hook);
        }
    }

    /**
     * Triggers Action Scheduler cleanup if enabled in the configuration.
     * 
     * This method checks if Action Scheduler cleanup is enabled in the configuration
     * and if so, executes the cleanup process.
     * 
     * @return void
     */
    public static function maybe_run_cleanup()
    {
        $config = SysConfig::$config;
        
        // If ActionScheduler cleanup is enabled, perform it
        if (isset($config['db_actionscheduler_clean']) && $config['db_actionscheduler_clean']) {
            self::cleanup();
        }
    }

    /**
     * Cleans up Action Scheduler tables by removing old entries.
     * 
     * This method removes old logs and completed/canceled/failed actions based on
     * the configured retention periods. It also optimizes the tables if configured.
     * 
     * @return array{
     *     logs_deleted: int,
     *     actions_deleted: int
     * } Information about the cleanup operation including counts of deleted records.
     */
    public static function cleanup()
    {
        global $wpdb;
        
        $config = SysConfig::$config;
        
        // Cleanup actionscheduler_logs table - entries older than X days
        $days_logs = isset($config['db_actionscheduler_logs_days']) ? intval($config['db_actionscheduler_logs_days']) : 1;
        $logs_deleted = $wpdb->query("
            DELETE FROM {$wpdb->prefix}actionscheduler_logs
            WHERE log_date_gmt < DATE_SUB(NOW(), INTERVAL {$days_logs} DAY)
        ");

        // Cleanup actionscheduler_actions table - completed, canceled, and failed actions older than X days
        $days_actions = isset($config['db_actionscheduler_actions_days']) ? intval($config['db_actionscheduler_actions_days']) : 7;
        $actions_deleted = $wpdb->query("
            DELETE FROM {$wpdb->prefix}actionscheduler_actions
            WHERE status IN ('complete', 'canceled', 'failed')
            AND (last_attempt_gmt < DATE_SUB(NOW(), INTERVAL {$days_actions} DAY) OR last_attempt_gmt IS NULL)
        ");

        // Optimalizace tabulek, pokud je povolena
        if (isset($config['db_optimize_tables']) && $config['db_optimize_tables']) {
            $wpdb->query("OPTIMIZE TABLE {$wpdb->prefix}actionscheduler_actions");
            $wpdb->query("OPTIMIZE TABLE {$wpdb->prefix}actionscheduler_logs");
        }

        return [
            'logs_deleted' => $logs_deleted !== false ? $logs_deleted : 0,
            'actions_deleted' => $actions_deleted !== false ? $actions_deleted : 0
        ];
    }

    /**
     * Retrieves information about the state of Action Scheduler tables.
     * 
     * This method collects various statistics about the Action Scheduler tables
     * including their sizes, record counts, and counts of actions by status.
     * 
     * @return array{
     *     actions_size: float|null,    // Size of actions table in MB
     *     logs_size: float|null,       // Size of logs table in MB
     *     logs_count: string,          // Total count of log entries
     *     action_counts: array{        // Count of actions by status
     *         complete: string,
     *         pending: string,
     *         canceled: string,
     *         failed: string
     *     }
     } Information about the Action Scheduler tables.
     */
    public static function get_status_info()
    {
        global $wpdb;
        
        // Získání velikosti tabulky actionscheduler_actions v MB
        $actions_size = $wpdb->get_var("
            SELECT ROUND((data_length + index_length) / 1024 / 1024, 2) 
            FROM information_schema.TABLES
            WHERE table_schema = '{$wpdb->dbname}'
            AND table_name = '{$wpdb->prefix}actionscheduler_actions'
        ");
        
        // Získání velikosti tabulky actionscheduler_logs v MB
        $logs_size = $wpdb->get_var("
            SELECT ROUND((data_length + index_length) / 1024 / 1024, 2) 
            FROM information_schema.TABLES
            WHERE table_schema = '{$wpdb->dbname}'
            AND table_name = '{$wpdb->prefix}actionscheduler_logs'
        ");
        
        // Získání počtu záznamů v tabulce actionscheduler_logs
        $logs_count = $wpdb->get_var("
            SELECT COUNT(*) as total_count
            FROM {$wpdb->prefix}actionscheduler_logs
        ");
        
        // Získání počtu záznamů pro každý stav akce
        $action_types = ['complete', 'pending', 'canceled', 'failed'];
        $action_counts = [];
        
        foreach ($action_types as $action_type) {
            $action_counts[$action_type] = $wpdb->get_var($wpdb->prepare("
                SELECT COUNT(*)
                FROM {$wpdb->prefix}actionscheduler_actions
                WHERE status = %s
            ", $action_type));
        }
        
        return [
            'actions_size' => $actions_size,
            'logs_size' => $logs_size,
            'logs_count' => $logs_count,
            'action_counts' => $action_counts
        ];
    }
}
