<?php
/**
 * ZiziCache Advanced Cache Drop-in
 *
 * Serves pre-generated HTML cache files directly, bypassing WordPress for eligible requests.
 */

$config = CONFIG_TO_REPLACE;

if (!headers_sent()) {
  // Send X-ZiziCache headers indicating cache status
  header('x-zizi-cache-cache: MISS [X]');
  header('x-zizi-cache-source: PHP');
}

// Skip caching for WP-CLI processes
if (defined('WP_CLI') && WP_CLI) {
  if (!headers_sent()) header('x-zizi-cache-cache: BYPASS [BYPASS]');
  return false;
}

// Bypass cache when 'cache_bust' query parameter is present
if (isset($_GET['cache_bust'])) {
  if (!headers_sent()) header('x-zizi-cache-cache: BYPASS [BYPASS]');
  return false;
}

// Only serve cache for GET and HEAD requests
if (!isset($_SERVER['REQUEST_METHOD']) || !in_array($_SERVER['REQUEST_METHOD'], ['HEAD', 'GET'])) {
  if (!headers_sent()) header('x-zizi-cache-cache: BYPASS [BYPASS]');
  return false;
}

// Skip REST API requests
if (strpos($_SERVER['REQUEST_URI'], '/wp-json/') !== false) {
  if (!headers_sent()) header('x-zizi-cache-cache: BYPASS [REST-API]');
  return false;
}

// Bypass cache when a configured bypass cookie is present
foreach ($config['cache_bypass_cookies'] as $cookie) {
  if (preg_grep("/$cookie/i", array_keys($_COOKIE))) {
    if (!headers_sent()) header('x-zizi-cache-cache: BYPASS [BYPASS]');
    return false;
  }
}

// If user is logged in and caching for authenticated users is disabled, bypass cache
$is_user_logged_in = preg_grep('/^wordpress_logged_in_/i', array_keys($_COOKIE));
if ($is_user_logged_in && !$config['cache_logged_in']) {
  if (!headers_sent()) header('x-zizi-cache-cache: BYPASS [BYPASS]');
  return false;
}

// Base filename 'index' when no specific path is provided
$file_name = 'index';

// Append '-logged-in' suffix for logged-in user cache variant
$file_name .= $is_user_logged_in ? '-logged-in' : '';

// Append user role(s) to cache filename for role-based caching
$file_name .= isset($_COOKIE['zc_logged_in_roles']) ? '-' . $_COOKIE['zc_logged_in_roles'] : '';

// Append Aelia currency code to cache filename if detected
$file_name .= isset($_COOKIE['aelia_cs_selected_currency'])
  ? '-' . $_COOKIE['aelia_cs_selected_currency']
  : '';

// Append YITH currency code to cache filename if detected
$file_name .= isset($_COOKIE['yith_wcmcs_currency']) ? '-' . $_COOKIE['yith_wcmcs_currency'] : '';

// Append WCML currency code to cache filename if detected
$file_name .= isset($_COOKIE['wcml_currency']) ? '-' . $_COOKIE['wcml_currency'] : '';

// Exclude ignored query parameters and hash remaining values for unique cache key
$query_strings = array_diff_key($_GET, array_flip($config['cache_ignore_queries']));
$file_name .= !empty($query_strings) ? '-' . md5(serialize($query_strings)) : '';

// Determine filesystem path for the cached file
$host = $_SERVER['HTTP_HOST'];
$path = parse_url($_SERVER['REQUEST_URI'], PHP_URL_PATH);
$path = urldecode($path);
$cache_file_path = WP_CONTENT_DIR . "/cache/zizi-cache/$host/$path/$file_name.html.gz";

// Return false if the gzipped cache file is not found
if (!file_exists($cache_file_path)) {
  if (!headers_sent()) header('x-zizi-cache-cache: MISS [X]');
  return false;
}

// --- Zizi Cache Logging ---
// Direct logging for early execution context
if (!function_exists('zizi_log_early')) {
    function zizi_log_early($message) {
        // Define unified log path directly
        $log_file = WP_CONTENT_DIR . '/plugins/zizi-cache/zizi-log.log';
        $date = date('Y-m-d H:i:s');
        $formatted_message = "[$date] [advanced-cache] $message"; // Add timestamp and context

        // Ensure directory exists (basic check)
        $log_dir = dirname($log_file);
        if (!is_dir($log_dir)) {
            @mkdir($log_dir, 0755, true);
        }

        // Append message
        @file_put_contents($log_file, $formatted_message . PHP_EOL, FILE_APPEND | LOCK_EX);
    }
}

// ---- DETEKCE WARMED CACHE ----
$fp = fopen($cache_file_path, 'rb');
$head = fread($fp, 4096); // přečte začátek souboru (většina podpisů bývá na začátku nebo konci, 4kB je bezpečné)
fclose($fp);
if (strpos($head, 'warmed') !== false) {
  if (!headers_sent()) header('x-zizi-cache-cache: WARMED');
} else {
  if (!headers_sent()) header('x-zizi-cache-cache: HIT [OK]');
}

// Disable PHP output compression to prevent conflicts
ini_set('zlib.output_compression', 0);

// Add headers for CDN and browser cache control
header("Cache-Tag: $host");
header('CDN-Cache-Control: max-age=2592000');
header('Cache-Control: no-cache, must-revalidate');

// ---- ZIZICACHE: SERVE STALE WHILE REVALIDATE + SQLITE ----
// Nastavení lifespan v sekundách (z configu nebo fallback)
$cache_expiry = isset($config['cache_lifespan']) ? ((int)$config['cache_lifespan'] * 3600) : 3600;
$cache_last_modified = filemtime($cache_file_path);
$is_expired = (time() - $cache_last_modified) > $cache_expiry;

if ($is_expired) {
  if (!headers_sent()) header('x-zizi-cache-cache: EXPIRED [EX]');
  // Vracíme stale cache, ale zároveň naplánujeme asynchronní obnovu
  try {
    $sqlite_path = WP_CONTENT_DIR . '/cache/zizi-cache/preload.sqlite';
    $db = new PDO('sqlite:' . $sqlite_path);
    $db->setAttribute(PDO::ATTR_ERRMODE, PDO::ERRMODE_EXCEPTION);
    $db->exec('CREATE TABLE IF NOT EXISTS zizi_expired_cache (
      url TEXT PRIMARY KEY,
      cache_path TEXT,
      expired_at INTEGER,
      status TEXT DEFAULT "pending"
    )');
    // Sestavení URL
    $scheme = (!empty($_SERVER['HTTPS']) && $_SERVER['HTTPS'] !== 'off') ? 'https' : 'http';
    $request_url = $scheme . '://' . $host . $path;
    $inserted = false;
    try {
      $stmt = $db->prepare('INSERT OR IGNORE INTO zizi_expired_cache (url, cache_path, expired_at, status) VALUES (?, ?, ?, "pending")');
      // Oprava: status je konstantní "pending", předáváme pouze 3 parametry
      $inserted = $stmt->execute([$request_url, $cache_file_path, time()]);
      $log_message = "URL zapis: $request_url - ".($inserted ? 'OK' : 'IGNORED');
      zizi_log_early($log_message);
    } catch (Exception $e) {
      $log_message = "URL zapis ERROR: $request_url - " . $e->getMessage();
      zizi_log_early($log_message);
    }
  } catch (Exception $e) {
    // Fail silently, fallback na stale cache
  }
}

// Send Last-Modified header based on cache file timestamp
$cache_last_modified = filemtime($cache_file_path);
header('Last-Modified: ' . gmdate('D, d M Y H:i:s', $cache_last_modified) . ' GMT');

// Retrieve 'If-Modified-Since' timestamp from client request
$http_modified_since = isset($_SERVER['HTTP_IF_MODIFIED_SINCE'])
  ? strtotime($_SERVER['HTTP_IF_MODIFIED_SINCE'])
  : 0;

// Send 304 Not Modified if client cache is up to date
if ($http_modified_since >= $cache_last_modified) {
  header($_SERVER['SERVER_PROTOCOL'] . ' 304 Not Modified', true, 304);
  exit();
}

// ---- OLS-AWARE HEADER MANAGEMENT ----
// Enhanced OpenLiteSpeed server detection using multiple methods
$is_ols_server = false;
$is_enterprise_ols = false;

// Method 1: Check LSWS_EDITION environment variable (most reliable)
if (isset($_SERVER['LSWS_EDITION'])) {
  $edition = strtolower($_SERVER['LSWS_EDITION']);
  if (strpos($edition, 'openlitespeed') !== false) {
    $is_ols_server = true;
  } elseif (strpos($edition, 'litespeed') !== false) {
    $is_enterprise_ols = true;
  }
}

// Method 2: Check SERVER_SOFTWARE header (fallback)
if (!$is_ols_server && !$is_enterprise_ols && isset($_SERVER['SERVER_SOFTWARE'])) {
  $server_software = strtolower($_SERVER['SERVER_SOFTWARE']);
  if (strpos($server_software, 'openlitespeed') !== false) {
    $is_ols_server = true;
  } elseif (strpos($server_software, 'litespeed') !== false) {
    $is_enterprise_ols = true;
  }
}

// Method 3: Check for LiteSpeed-specific environment variables
if (!$is_ols_server && !$is_enterprise_ols) {
  if (isset($_SERVER['LSPHP_ENABLE']) || isset($_SERVER['LS_WORDPRESS']) || 
      isset($_SERVER['LSAPI_CHILDREN']) || isset($_SERVER['LSAPI_AVOID_FORK'])) {
    $is_ols_server = true; // Assume OLS if LiteSpeed PHP handler is present
  }
}

// Set appropriate headers based on server type
if ($is_ols_server || $is_enterprise_ols) {
  // Ultra-aggressive header handling for OpenLiteSpeed
  header('Content-Type: text/html; charset=UTF-8', true);
  header('Content-Encoding: gzip', true);
  header('Content-Disposition: inline', true);
  header('X-Content-Type-Options: nosniff', true);
  header('Cache-Control: public, max-age=31536000', true);
  header('Vary: Accept-Encoding', true);
  header('X-ZC-OLS-Override: true', true);
} else {
  // Standard headers for other servers
  header('Content-Type: text/html; charset=UTF-8', true);
  header('Content-Encoding: gzip', true);
  header('Cache-Control: public, max-age=31536000', true);
  header('Vary: Accept-Encoding', true);
  header('Content-Disposition: inline', true);
}

// Read and output the cache file
readfile($cache_file_path);
exit();
