/**
 * ZiziCache LCP Detector
 * 
 * Detekce Largest Contentful Paint (LCP) elementů a ukládání informací o nich
 * s rozlišením mezi mobilními a desktopovými zařízeními.
 * 
 * Využívá PerformanceObserver a Layout Instability API pro identifikaci LCP.
 */
(function() {
    // Neběžet v admin seckcích
    if (document.body && document.body.classList.contains('wp-admin')) {
        return;
    }

    // Inicializační log - potvrzení, že skript byl načten
    console.log('🚀 ZiziCache LCP Detector: Initialized');
    
    // Kontrola konfigurace - důležité pro debugování
    if (!window.ZiziCacheConfig) {
        console.warn('⚠️ ZiziCache LCP Detector: Konfigurace chybí!');
    } else {
        console.log('✅ ZiziCache LCP Detector: Konfigurace načtena', {
            isAdmin: window.ZiziCacheConfig.isAdmin,
            debugMode: window.ZiziCacheConfig.debugMode
        });
    }

    // Detekce typu zařízení
    const isMobile = window.innerWidth < 768;
    const deviceType = isMobile ? 'mobile' : 'desktop';
    
    // Informace o LCP elementu
    let lcpInfo = {
        url: window.location.pathname,
        deviceType: deviceType,
        timestamp: Date.now(),
        element: null,
        selector: null,
        src: null,
        size: 0,
        renderTime: 0
    };    // OPRAVA: Úplně odstraněno duplicitní nastavení isAdmin
    // Namísto toho se na všech místech bude kontrolovat přímo window.ZiziCacheConfig.isAdmin
    console.debug('ZiziCache LCP Detector: Initializing', { 
        config: window.ZiziCacheConfig,
        hasConfig: !!window.ZiziCacheConfig
    });    // Zvýrazní LCP element na stránce pro administrátory nebo pokud je povoleno veřejné zobrazení
    // NOVÝ PŘÍSTUP: Použijeme overlay přes celou stránku místo modifikace DOM struktury
    function highlightLcpElement(element) {
        if (!element) return;
        
        console.log('ZiziCache: Pokus o zvýraznění LCP elementu', { 
            element: element, 
            tagName: element.tagName,
            config: window.ZiziCacheConfig,
            isWooProduct: isWooCommerceProduct()
        });
        
        try {
            // Přidáme pouze CSS třídu pro zvýraznění, bez změny struktury DOM
            element.classList.add('zizicache-lcp-element');
            
            // Získáme pozici a velikost elementu na stránce
            const rect = element.getBoundingClientRect();
            const scrollX = window.pageXOffset || document.documentElement.scrollLeft;
            const scrollY = window.pageYOffset || document.documentElement.scrollTop;
            
            // Vytvoříme odznak jako fixní element, který nebude součástí layoutu
            const badge = document.createElement('div');
            badge.className = 'zizicache-lcp-badge';
            badge.textContent = `LCP: ${Math.round(lcpInfo.renderTime)}ms`;
            badge.style.position = 'absolute';
            badge.style.zIndex = '2147483647';  // Maximální hodnota z-indexu
            badge.style.backgroundColor = '#22c55e';
            badge.style.color = 'white';
            badge.style.fontSize = '11px';
            badge.style.fontWeight = 'bold';
            badge.style.padding = '3px 6px';
            badge.style.borderRadius = '3px';
            badge.style.boxShadow = '0 1px 3px rgba(0,0,0,0.3)';
            badge.style.pointerEvents = 'none';
            badge.style.whiteSpace = 'nowrap';
            badge.style.lineHeight = '1';
            badge.style.fontFamily = 'Arial, sans-serif';
            
            // Přidáme informaci o typu elementu
            const isBackgroundImage = element.tagName !== 'IMG' && lcpInfo.src;
            if (element.tagName === 'IMG') {
                badge.textContent += ' [IMG]';
            } else if (isBackgroundImage) {
                badge.textContent += ' [BG-IMG]';
            } else {
                badge.textContent += ` [${element.tagName}]`;
            }
            
            // Umístíme odznak na vrch elementu
            badge.style.top = (rect.top + scrollY - 18) + 'px';
            badge.style.left = (rect.left + scrollX) + 'px';
            
            // Přidáme odznak do těla dokumentu (mimo DOM strukturu elementu)
            document.body.appendChild(badge);
            
            // Vytvoříme ohraničení kolem elementu (zvlášť pro obrázky na pozadí)
            if (isBackgroundImage) {
                // Pro elementy s obrázkem na pozadí použijeme speciální ohraničení
                const outline = document.createElement('div');
                outline.className = 'zizicache-lcp-outline';
                outline.style.position = 'absolute';
                outline.style.zIndex = '2147483646'; // O jeden méně než odznak
                outline.style.border = '3px dashed #22c55e';
                outline.style.pointerEvents = 'none';
                outline.style.boxSizing = 'border-box';
                outline.style.top = (rect.top + scrollY) + 'px';
                outline.style.left = (rect.left + scrollX) + 'px';
                outline.style.width = rect.width + 'px';
                outline.style.height = rect.height + 'px';
                
                document.body.appendChild(outline);
            }
            
            // Dbáme na to, aby se ohraničení a odznak aktualizovaly při scrollování
            const updateBadgePosition = () => {
                const newRect = element.getBoundingClientRect();
                const newScrollX = window.pageXOffset || document.documentElement.scrollLeft;
                const newScrollY = window.pageYOffset || document.documentElement.scrollTop;
                
                badge.style.top = (newRect.top + newScrollY - 18) + 'px';
                badge.style.left = (newRect.left + newScrollX) + 'px';
                
                if (isBackgroundImage) {
                    const outline = document.querySelector('.zizicache-lcp-outline');
                    if (outline) {
                        outline.style.top = (newRect.top + newScrollY) + 'px';
                        outline.style.left = (newRect.left + newScrollX) + 'px';
                        outline.style.width = newRect.width + 'px';
                        outline.style.height = newRect.height + 'px';
                    }
                }
            };
            
            // Přidáme event listener na scroll a resize
            window.addEventListener('scroll', updateBadgePosition);
            window.addEventListener('resize', updateBadgePosition);
            
            // Pro debugging
            console.log('ZiziCache: LCP element zvýrazněn', {
                element: element,
                tagName: element.tagName,
                size: Math.round(lcpInfo.size / 1000) + ' KB',
                renderTime: Math.round(lcpInfo.renderTime) + ' ms',
                rect: {
                    top: rect.top,
                    left: rect.left,
                    width: rect.width,
                    height: rect.height
                }
            });
        } catch (error) {
            console.error('ZiziCache: Chyba při zvýrazňování LCP elementu', error);
        }
    }

    // Speciální funkce pro WooCommerce elementy - kontrola jestli jsme na stránce produktu
    function isWooCommerceProduct() {
        // Kontrola URL pro WooCommerce
        return window.location.pathname.includes('/produkt/') || 
               document.body.classList.contains('woocommerce-product-page') ||
               document.body.classList.contains('single-product');
    }

    // Pozorovatel pro LCP
    const lcpObserver = new PerformanceObserver((entryList) => {
        const entries = entryList.getEntries();
        const lastEntry = entries[entries.length - 1];
        
        if (!lastEntry) return;
        
        // Uchovávej jen největší LCP element
        if (lastEntry.size > lcpInfo.size) {
            const element = lastEntry.element;
            if (!element) return;
            
            let selector = '';
            let src = '';
            
            // Pokus se určit unikátní selektor pro element
            if (element.id) {
                selector = `#${element.id}`;
            } else if (element.classList && element.classList.length) {
                selector = `.${Array.from(element.classList).join('.')}`;
            } else {
                // Vytvoř relativní cestu k elementu
                let path = [];
                let currentElem = element;
                while (currentElem && currentElem !== document.body) {
                    let tag = currentElem.tagName.toLowerCase();
                    if (currentElem.id) {
                        path.unshift(`${tag}#${currentElem.id}`);
                        break;
                    } else {
                        let index = 0;
                        let sibling = currentElem;
                        while (sibling) {
                            if (sibling.tagName && sibling.tagName.toLowerCase() === tag) {
                                index++;
                            }
                            sibling = sibling.previousElementSibling;
                        }
                        path.unshift(`${tag}:nth-of-type(${index})`);
                    }
                    currentElem = currentElem.parentElement;
                }
                selector = path.join(' > ');
            }
            
            // Informace o lazy loadingu a fetch priority pro obrázky
            let lazyLoadInfo = '';
            let fetchPriorityInfo = '';
            let loadingAttr = '';
            
            // Pokud jde o obrázek, získej jeho URL a atributy
            if (element.tagName === 'IMG') {
                src = element.src || '';
                loadingAttr = element.getAttribute('loading') || 'auto';
                fetchPriorityInfo = element.getAttribute('fetchpriority') || 'auto';
                
                // Detekce lazy loadingu
                if (loadingAttr === 'lazy') {
                    lazyLoadInfo = 'lazy';
                } else if (loadingAttr === 'eager') {
                    lazyLoadInfo = 'eager';
                }
            } else if (element.tagName === 'IMAGE') {
                src = element.getAttribute('href') || '';
            } else {
                // Zkontroluj background-image
                const style = window.getComputedStyle(element);
                const bgImage = style.backgroundImage;
                if (bgImage && bgImage !== 'none') {
                    const match = bgImage.match(/url\(['"]?([^'"()]+)['"]?\)/);
                    if (match && match[1]) {
                        src = match[1];
                    }
                }
            }
            
            // Aktualizuj informace o LCP
            lcpInfo.element = element.tagName.toLowerCase();
            lcpInfo.selector = selector;
            lcpInfo.src = src;
            lcpInfo.size = lastEntry.size;
            lcpInfo.renderTime = lastEntry.renderTime || lastEntry.loadTime;
            lcpInfo.loadingAttr = loadingAttr;
            lcpInfo.fetchPriority = fetchPriorityInfo;            // DŮLEŽITÁ OPRAVA: Vždy zobrazíme debug info, ale zkontrolujeme pouze existenci ZiziCacheConfig
            // Toto opraví problém s false pozitivním porovnáním isAdmin === true
            const hasConfig = !!window.ZiziCacheConfig;
            const showDebugInfo = hasConfig && window.ZiziCacheConfig.isAdmin;
            
            // Debug pro sledování hodnoty ZiziCacheConfig
            console.debug('ZiziCache LCP Debug:', {
                hasConfig: hasConfig,
                config: window.ZiziCacheConfig,
                showDebugInfo: showDebugInfo,
                element: element,
                tagName: element.tagName,
                elementSize: Math.round(lcpInfo.size / 1000) + ' KB',
                computedStyle: {
                    position: window.getComputedStyle(element).position,
                    display: window.getComputedStyle(element).display,
                    zIndex: window.getComputedStyle(element).zIndex
                },
                parentInfo: element.parentNode ? {
                    tagName: element.parentNode.tagName,
                    position: window.getComputedStyle(element.parentNode).position,
                    display: window.getComputedStyle(element.parentNode).display
                } : 'No parent'
            });
            
            // Zvýrazni LCP element pokud konfigurace existuje a umožňuje to
            if (showDebugInfo) {
                console.log('🟢 ZiziCache: Zvýrazňuji LCP element', element);
                highlightLcpElement(element);
            } else {
                console.debug('⚪ ZiziCache: LCP element nebude zvýrazněn (nemáte oprávnění nebo není povoleno)');
            }
            
            // Detailnější výpis o LCP do konzole - pouze pokud je povoleno
            if (showDebugInfo) {
                console.group('🔍 ZiziCache: Detekován LCP element');
                console.log(`%c📊 Typ elementu: %c${lcpInfo.element}`, 'font-weight: bold', 'color: blue; font-weight: bold');
                
                if (lcpInfo.element === 'img') {
                    console.log(`%c🖼️ LCP je OBRÁZEK`, 'font-weight: bold', 'color: green; font-weight: bold');
                    console.log(`%c📏 Velikost: %c${Math.round(lcpInfo.size / 1000)} KB`, 'font-weight: bold', 'color: purple');
                    console.log(`%c⏱️ Čas vykreslení: %c${Math.round(lcpInfo.renderTime)} ms`, 'font-weight: bold', 'color: orange');
                    
                    // Informace o načítání
                    if (loadingAttr) {
                        console.log(`%c🔄 Loading atribut: %c${loadingAttr}`, 'font-weight: bold', loadingAttr === 'lazy' ? 'color: red; font-weight: bold' : 'color: green; font-weight: bold');
                    }
                    
                    // Informace o fetch priority
                    if (fetchPriorityInfo) {
                        console.log(`%c🚀 Fetch Priority: %c${fetchPriorityInfo}`, 'font-weight: bold', fetchPriorityInfo === 'high' ? 'color: green; font-weight: bold' : 'color: orange');
                    }
                } else if (lcpInfo.src) {
                    // Pro elementy s pozadím
                    console.log(`%c🖼️ LCP je element s obrázkem na pozadí`, 'font-weight: bold', 'color: green; font-weight: bold');
                    console.log(`%c📏 Velikost: %c${Math.round(lcpInfo.size / 1000)} KB`, 'font-weight: bold', 'color: purple');
                    console.log(`%c⏱️ Čas vykreslení: %c${Math.round(lcpInfo.renderTime)} ms`, 'font-weight: bold', 'color: orange');
                } else {
                    // Pro textové elementy
                    console.log(`%c📝 LCP je TEXTOVÝ element: %c${selector}`, 'font-weight: bold', 'color: red; font-weight: bold');
                    console.log(`%c📄 Obsah: %c${element.textContent?.substring(0, 50)}${element.textContent?.length > 50 ? '...' : ''}`, 'font-weight: bold', 'color: gray');
                    console.log(`%c📏 Velikost: %c${Math.round(lcpInfo.size / 1000)} KB`, 'font-weight: bold', 'color: purple');
                    console.log(`%c⏱️ Čas vykreslení: %c${Math.round(lcpInfo.renderTime)} ms`, 'font-weight: bold', 'color: orange');
                }
                
                console.log(`%c🎯 CSS Selektor: %c${selector}`, 'font-weight: bold', 'color: brown');
                
                console.groupEnd();
            }
            
            // Ulož informace do localStorage
            saveLcpInfo();
        }
    });

    // Funkce pro uložení informací o LCP
    function saveLcpInfo() {
        try {
            // Ukládání do lokálního úložiště
            const storageKey = `zizicache_lcp_info_${deviceType}`;
            
            // Zkontroluj, jestli je již něco uloženo
            const existingDataStr = localStorage.getItem(storageKey);
            let existingData = [];
            
            if (existingDataStr) {
                try {
                    existingData = JSON.parse(existingDataStr);
                    if (!Array.isArray(existingData)) {
                        existingData = [];
                    }
                } catch (e) {
                    console.error('ZiziCache LCP Detector: Chyba parsování dat z localStorage', e);
                    existingData = [];
                }
            }
            
            // Najdi existující záznam pro tuto URL
            const existingIndex = existingData.findIndex(item => item.url === lcpInfo.url);
            
            if (existingIndex >= 0) {
                // Aktualizuj existující záznam
                existingData[existingIndex] = lcpInfo;
            } else {
                // Přidej nový záznam
                existingData.push(lcpInfo);
                
                // Omez počet záznamů na 50
                if (existingData.length > 50) {
                    existingData.shift(); // Odstraň nejstarší záznam
                }
            }
            
            localStorage.setItem(storageKey, JSON.stringify(existingData));
            
            // Po uložení odešli data na server pomocí Beacon API
            sendLcpInfoToServer();
        } catch (e) {
            console.error('ZiziCache LCP Detector: Chyba při ukládání dat', e);
        }
    }

    // Odeslání dat na server
    function sendLcpInfoToServer() {
        // Pokud máme WordPress REST API endpoint v konfiguraci
        if (window.ZiziCacheConfig && window.ZiziCacheConfig.api) {
            const data = {
                lcpInfo: lcpInfo,
                nonce: window.ZiziCacheConfig.api.nonce || ''
            };
            
            const jsonData = JSON.stringify(data);
              // Získáme aktuální hodnotu pro zobrazování debugovacích informací
            const showDebugInfo = window.ZiziCacheConfig && window.ZiziCacheConfig.isAdmin;
            
            // Zkusíme poslat data na primární endpoint
            sendDataToEndpoint(window.ZiziCacheConfig.api.lcpEndpoint, jsonData)
                .catch(error => {
                    if (showDebugInfo) {
                        console.warn('ZiziCache LCP Detector: Primární endpoint selhal, zkouším fallback endpointy', error);
                    }
                    
                    // Pokud máme fallback endpointy, zkusíme je postupně
                    if (window.ZiziCacheConfig.api.fallbackEndpoints && Array.isArray(window.ZiziCacheConfig.api.fallbackEndpoints)) {
                        let endpointPromises = [];
                        
                        // Začínáme od druhého (index 1), protože první jsme už zkusili
                        for (let i = 1; i < window.ZiziCacheConfig.api.fallbackEndpoints.length; i++) {
                            endpointPromises.push(
                                sendDataToEndpoint(window.ZiziCacheConfig.api.fallbackEndpoints[i], jsonData)
                                    .then(() => {
                                        if (showDebugInfo) {
                                            console.log('ZiziCache LCP Detector: Data úspěšně odeslána přes fallback endpoint', 
                                                window.ZiziCacheConfig.api.fallbackEndpoints[i]);
                                        }
                                        return true; // Uspěli jsme
                                    })
                                    .catch(() => false) // Neuspěli jsme
                            );
                        }
                        
                        // Zkusíme všechny fallback endpointy
                        Promise.all(endpointPromises).then((results) => {
                            // Pokud žádný endpoint nebyl úspěšný, zkusíme AJAX jako poslední možnost
                            if (!results.includes(true) && window.ZiziCacheConfig.api.ajaxUrl) {
                                sendDataViaAjax(window.ZiziCacheConfig.api.ajaxUrl, lcpInfo);
                            }
                        });
                    } else if (window.ZiziCacheConfig.api.ajaxUrl) {
                        // Pokud nemáme fallback endpointy, zkusíme přímo AJAX
                        sendDataViaAjax(window.ZiziCacheConfig.api.ajaxUrl, lcpInfo);
                    }
                });
        }
    }
      // Funkce pro odeslání dat přes WordPress AJAX
    function sendDataViaAjax(ajaxUrl, lcpInfo) {
        // Získáme aktuální hodnotu pro zobrazování debugovacích informací
        const showDebugInfo = window.ZiziCacheConfig && window.ZiziCacheConfig.isAdmin;
        
        if (showDebugInfo) {
            console.log('ZiziCache LCP Detector: Zkouším poslední možnost - WordPress AJAX');
        }
        
        const formData = new FormData();
        formData.append('action', 'zizi_lcp_data'); // WordPress AJAX action
        formData.append('lcpInfo', JSON.stringify(lcpInfo));
        
        fetch(ajaxUrl, {
            method: 'POST',
            credentials: 'same-origin',
            body: formData
        })
        .then(response => {
            if (response.ok && showDebugInfo) {
                console.log('ZiziCache LCP Detector: Data úspěšně odeslána přes AJAX');
            } else if (!response.ok && showDebugInfo) {
                console.error('ZiziCache LCP Detector: Chyba AJAX', response.statusText);
            }
        })
        .catch(error => {
            if (showDebugInfo) {
                console.error('ZiziCache LCP Detector: Chyba AJAX', error);
            }
        });
    }

    // Funkce pro odeslání dat na konkrétní endpoint
    function sendDataToEndpoint(endpoint, jsonData) {
        return new Promise((resolve, reject) => {
            try {
                fetch(endpoint, {
                    method: 'POST',
                    credentials: 'same-origin',
                    headers: {
                        'Content-Type': 'application/json',
                    },
                    body: jsonData
                })
                .then(response => {
                    if (response.ok) {
                        resolve(response);
                    } else {
                        reject(new Error(`Endpoint ${endpoint} vrátil status ${response.status}`));
                    }
                })                .catch(error => {
                    // Získáme aktuální hodnotu pro zobrazování debugovacích informací
                    const showDebugInfo = window.ZiziCacheConfig && window.ZiziCacheConfig.isAdmin;
                    
                    if (showDebugInfo) {
                        console.error(`ZiziCache LCP Detector: Chyba při odesílání dat na endpoint ${endpoint}`, error);
                    }
                    
                    // Zkusíme alternativní metody odeslání
                    try {
                        // Zkusit Beacon API jako fallback
                        if (navigator.sendBeacon) {
                            const success = navigator.sendBeacon(endpoint, jsonData);
                            if (success) {
                                resolve();
                                return;
                            }
                        }
                        
                        // Poslední možnost - XMLHttpRequest
                        const xhr = new XMLHttpRequest();
                        xhr.open('POST', endpoint, true);
                        xhr.setRequestHeader('Content-Type', 'application/json');
                        xhr.onload = function() {
                            if (xhr.status >= 200 && xhr.status < 300) {
                                resolve();
                            } else {
                                reject(new Error(`XHR Error: ${xhr.statusText}`));
                            }
                        };
                        xhr.onerror = function() {
                            reject(new Error('XHR Request failed'));
                        };
                        xhr.send(jsonData);
                    } catch(e) {
                        reject(e);
                    }
                });
            } catch(e) {
                reject(e);
            }
        });
    }    // Spustí sledování LCP
    function observeLCP() {
        // Log to confirm the observer is starting
        console.debug('ZiziCache LCP Detector: Starting LCP observation');
        
        lcpObserver.observe({type: 'largest-contentful-paint', buffered: true});
        
        // Zobrazit statistiky o zaznamenaných LCP elementech pro adminy nebo když je povoleno veřejné zobrazení
        const showDebugInfo = window.ZiziCacheConfig && window.ZiziCacheConfig.isAdmin === true;
        if (showDebugInfo) {
            showLcpStats();
        }
    }
    
    // Zobrazí statistiky o zaznamenaných LCP elementech
    function showLcpStats() {
        // Check again if we should show stats (for consistency)
        const showDebugInfo = window.ZiziCacheConfig && window.ZiziCacheConfig.isAdmin === true;
        if (!showDebugInfo) return; // Pouze pokud je povoleno
        
        try {
            const storageKey = `zizicache_lcp_info_${deviceType}`;
            const existingDataStr = localStorage.getItem(storageKey);
            
            if (!existingDataStr) return;
            
            const existingData = JSON.parse(existingDataStr);
            if (!Array.isArray(existingData) || existingData.length === 0) return;
            
            // Statistiky podle typu elementu
            const elementTypes = {};
            const imageCount = existingData.filter(item => item.element === 'img').length;
            const textCount = existingData.filter(item => !item.src).length;
            const bgImageCount = existingData.filter(item => item.element !== 'img' && item.src).length;
            
            console.group('📊 ZiziCache: Statistiky LCP elementů');
            console.log(`%cPočet zaznamenaných LCP elementů: %c${existingData.length}`, 'font-weight: bold', 'color: blue; font-weight: bold');
            console.log(`%c🖼️ Obrázky (IMG): %c${imageCount} (${Math.round(imageCount/existingData.length*100)}%)`, 'font-weight: bold', imageCount > 0 ? 'color: green; font-weight: bold' : '');
            console.log(`%c🎨 Elementy s obrázkem na pozadí: %c${bgImageCount} (${Math.round(bgImageCount/existingData.length*100)}%)`, 'font-weight: bold', bgImageCount > 0 ? 'color: orange; font-weight: bold' : '');
            console.log(`%c📝 Textové elementy: %c${textCount} (${Math.round(textCount/existingData.length*100)}%)`, 'font-weight: bold', textCount > 0 ? 'color: red; font-weight: bold' : '');
            
            if (imageCount > 0) {
                // Vypsat nejčastější LCP obrázky
                const images = existingData.filter(item => item.element === 'img');
                const imageSources = {};
                
                images.forEach(item => {
                    const srcBase = item.src?.split('?')[0] || 'unknown';
                    imageSources[srcBase] = (imageSources[srcBase] || 0) + 1;
                });
                
                const topImages = Object.entries(imageSources)
                    .sort((a, b) => b[1] - a[1])
                    .slice(0, 3);
                
                if (topImages.length > 0) {
                    console.log('%c🔝 Nejčastější LCP obrázky:', 'font-weight: bold');
                    topImages.forEach(([src, count], index) => {
                        console.log(`%c   ${index + 1}. %c${src} %c(${count}x)`, 'font-weight: bold', 'color: green', 'color: gray');
                    });
                }
            }
            
            console.groupEnd();
        } catch (e) {
            console.error('ZiziCache LCP Detektor: Chyba při zobrazování statistik', e);
        }
    }

    // Spusť po načtení stránky
    if (document.readyState === 'complete') {
        observeLCP();
    } else {
        window.addEventListener('load', observeLCP);
    }

    // Očisti po ukončení sledování
    window.addEventListener('beforeunload', () => {
        lcpObserver.disconnect();
    });
})();
