/**
 * ZiZi Cache - User Interaction Handler
 * 
 * Tento skript zpracovává odložené JavaScripty, které se spustí až po první
 * interakci uživatele s webovou stránkou. Zlepšuje výkon stránky tím, že
 * odkládá neesenciální scripty až do doby, kdy jsou skutečně potřeba.
 * 
 * Funkce:
 * - Detekuje uživatelskou interakci (mousedown, mousemove, keydown, scroll, touchstart)
 * - Spouští odložené scripty po první interakci
 * - Má fallback timeout pro případy bez interakce
 * - Přizpůsobuje se rychlosti sítě (3G, 4G, 5G+)
 * - Obsahuje bezpečnostní validace
 * 
 * @since 1.0.0
 * @author ZiZi Cache Plugin
 */

(function() {
  "use strict";
  
  // Configuration - will be injected by PHP
  const TIMEOUT = window.ziziUserInteractionConfig?.timeout || 3000;
  const DEBUG = window.ziziUserInteractionConfig?.debug || false;
  
  // State management
  let isTriggered = false;
  let delayedScripts = [];
  let processedCount = 0;
  
  // Network awareness (updated for 2025 - 2G is practically extinct)
  let networkSpeed = "fast";
  if ("connection" in navigator && navigator.connection) {
    const conn = navigator.connection;
    if (conn.effectiveType === "3g") {
      networkSpeed = "slow";
    } else if (conn.effectiveType === "4g") {
      networkSpeed = "medium";
    }
    // 5G and faster connections remain "fast"
  }
  
  /**
   * Secure logging function
   * @param {string} message - Log message
   * @param {*} data - Optional data to log
   */
  function log(message, data) {
    if (DEBUG && console && console.log) {
      console.log("[ZiZi User Interaction Delay]", message, data || "");
    }
  }
  
  /**
   * Collect delayed scripts with validation
   * Hledá všechny scripty s atributem data-zizi-delay
   */
  function collectDelayedScripts() {
    const scripts = document.querySelectorAll("script[data-zizi-delay]");
    delayedScripts = Array.from(scripts).filter(script => {
      // Additional validation for security
      return script.hasAttribute("data-zizi-delay") && 
             !script.hasAttribute("data-zizi-processed");
    });
    log("Found delayed scripts:", delayedScripts.length);
  }
  
  /**
   * Execute script with security checks
   * @param {HTMLScriptElement} scriptElement - Script element to execute
   */
  function executeScript(scriptElement) {
    if (scriptElement.hasAttribute("data-zizi-processed")) {
      return;
    }
    
    scriptElement.setAttribute("data-zizi-processed", "true");
    
    try {
      const newScript = document.createElement("script");
      
      // Copy attributes except delay-specific ones
      Array.from(scriptElement.attributes).forEach(attr => {
        if (!attr.name.startsWith("data-zizi-delay")) {
          newScript.setAttribute(attr.name, attr.value);
        }
      });
      
      // Handle external script
      if (scriptElement.hasAttribute("data-zizi-delay-src")) {
        const src = scriptElement.getAttribute("data-zizi-delay-src");
        // Basic URL validation
        if (src && (src.startsWith("http") || src.startsWith("/") || src.startsWith("./"))) {
          newScript.src = src;
          newScript.async = true;
        } else {
          log("Invalid script src:", src);
          return;
        }
      }
      
      // Handle inline script with security
      if (scriptElement.hasAttribute("data-zizi-delay-inline")) {
        const encodedContent = scriptElement.getAttribute("data-zizi-delay-inline");
        try {
          const decodedContent = atob(encodedContent);
          // Basic content validation
          if (decodedContent.length > 0 && decodedContent.length < 50000) {
            newScript.textContent = decodedContent;
          } else {
            log("Invalid inline script content length:", decodedContent.length);
            return;
          }
        } catch (e) {
          log("Failed to decode inline script:", e);
          return;
        }
      }
      
      // Insert and execute
      const parent = scriptElement.parentNode;
      if (parent) {
        parent.insertBefore(newScript, scriptElement);
        parent.removeChild(scriptElement);
        processedCount++;
        log("Executed script " + processedCount + "/" + delayedScripts.length);
      }
      
    } catch (error) {
      log("Error executing script:", error);
    }
  }
  
  /**
   * Execute all delayed scripts
   * Spustí všechny odložené scripty s ohledem na rychlost sítě
   */
  function executeDelayedScripts() {
    if (isTriggered) return;
    isTriggered = true;
    
    log("Executing delayed scripts...", {
      count: delayedScripts.length,
      networkSpeed: networkSpeed,
      trigger: "user-interaction"
    });
    
    // Network-aware execution
    if (networkSpeed === "slow" && delayedScripts.length > 3) {
      // Slow network: execute scripts with intervals
      let index = 0;
      const interval = setInterval(() => {
        if (index >= delayedScripts.length) {
          clearInterval(interval);
          return;
        }
        executeScript(delayedScripts[index]);
        index++;
      }, 150);
    } else {
      // Fast network: execute all scripts quickly
      delayedScripts.forEach(executeScript);
    }
  }
  
  /**
   * Setup event listeners for user interaction detection
   */
  function setupEventListeners() {
    const events = ["mousedown", "mousemove", "keydown", "scroll", "touchstart"];
    
    function onUserInteraction() {
      events.forEach(event => {
        document.removeEventListener(event, onUserInteraction, {passive: true});
      });
      
      // Small delay to ensure smooth interaction
      if ("requestIdleCallback" in window) {
        requestIdleCallback(executeDelayedScripts, {timeout: 100});
      } else {
        setTimeout(executeDelayedScripts, 50);
      }
    }
    
    events.forEach(event => {
      document.addEventListener(event, onUserInteraction, {passive: true});
    });
    
    log("Event listeners registered for events:", events);
  }
  
  /**
   * Fallback timeout with validation
   * Záložní mechanismus pro spuštění scriptů i bez interakce
   */
  function setupFallbackTimeout() {
    const validTimeout = Math.max(1000, Math.min(10000, TIMEOUT));
    setTimeout(() => {
      if (!isTriggered) {
        log("Fallback timeout triggered after " + validTimeout + "ms");
        executeDelayedScripts();
      }
    }, validTimeout);
  }
  
  /**
   * Initialize with error handling
   * Hlavní inicializační funkce
   */
  function init() {
    try {
      if (document.readyState === "loading") {
        document.addEventListener("DOMContentLoaded", () => {
          collectDelayedScripts();
          if (delayedScripts.length > 0) {
            setupEventListeners();
            setupFallbackTimeout();
            log("Initialized with " + delayedScripts.length + " delayed scripts");
          } else {
            log("No delayed scripts found, skipping initialization");
          }
        });
      } else {
        collectDelayedScripts();
        if (delayedScripts.length > 0) {
          setupEventListeners();
          setupFallbackTimeout();
          log("Initialized with " + delayedScripts.length + " delayed scripts");
        } else {
          log("No delayed scripts found, skipping initialization");
        }
      }
    } catch (error) {
      log("Initialization error:", error);
    }
  }
  
  // Start the magic
  init();
})();
