/**
 * ZiziCache Self-Host Enhanced JavaScript Library v2.0.0
 * Enhanced font loading and Google Fonts handling
 */
(function() {
'use strict';

// Configuration
var FONTS_CONFIG = {
    preconnectOrigins: [
        'https://fonts.googleapis.com',
        'https://fonts.gstatic.com',
        'https://use.typekit.net',
        'https://p.typekit.net'
    ],
    display: 'swap', // Default font-display value
    maxFontLoadWait: 3000, // Maximum time to wait for font loading
    criticalFonts: [], // Fonts to load immediately
    deferredFonts: [] // Fonts to load after user interaction
};

// Font loading optimization
var fontLoadingState = {
    initiated: false,
    preconnected: false,
    interactionTriggered: false
};

/**
 * Add preconnect links for font providers
 * Enhanced with crossorigin and improved timing
 */
function addFontPreconnects() {
    if (fontLoadingState.preconnected) return;
    
    FONTS_CONFIG.preconnectOrigins.forEach(function(origin) {
        // Check if preconnect already exists
        var existingPreconnect = document.querySelector('link[rel="preconnect"][href="' + origin + '"]');
        if (!existingPreconnect) {
            var preconnectLink = document.createElement('link');
            preconnectLink.rel = 'preconnect';
            preconnectLink.href = origin;
            preconnectLink.crossOrigin = 'anonymous';
            document.head.insertBefore(preconnectLink, document.head.firstChild);
        }
    });
    
    fontLoadingState.preconnected = true;
}

/**
 * Load critical fonts immediately with font-display: swap
 * Enhanced with better error handling and performance monitoring
 */
function loadCriticalFonts() {
    // Add preconnects first
    addFontPreconnects();
    
    // Find critical Google Fonts links
    var fontLinks = document.querySelectorAll('link[href*="fonts.googleapis.com"]');
    fontLinks.forEach(function(link) {
        // Check if it's marked as critical or if it's above the fold
        var isCritical = link.hasAttribute('data-critical') || 
                        link.getAttribute('media') === 'print' ||
                        !link.hasAttribute('data-defer');
        
        if (isCritical) {
            // Add font-display=swap to URL if not present
            var href = link.href;
            if (href.indexOf('display=') === -1) {
                href += (href.indexOf('?') === -1 ? '?' : '&') + 'display=swap';
                link.href = href;
            }
            
            // Remove media attribute for immediate loading
            if (link.media === 'print') {
                link.media = 'all';
            }
        }
    });
}

/**
 * Load deferred fonts after user interaction
 * Enhanced with intersection observer for better performance
 */
function loadDeferredFonts() {
    if (fontLoadingState.interactionTriggered) return;
    fontLoadingState.interactionTriggered = true;
    
    // Load deferred Google Fonts
    var deferredFontLinks = document.querySelectorAll('link[data-defer][href*="fonts.googleapis.com"]');
    deferredFontLinks.forEach(function(link) {
        // Add font-display=swap for better performance
        var href = link.href;
        if (href.indexOf('display=') === -1) {
            href += (href.indexOf('?') === -1 ? '?' : '&') + 'display=swap';
            link.href = href;
        }
        
        // Convert from print to all media
        if (link.media === 'print') {
            link.media = 'all';
        }
        
        // Remove defer attribute
        link.removeAttribute('data-defer');
    });
    
    // Load any inline font-face declarations
    loadInlineFontFaces();
    
    // Trigger font load event
    document.dispatchEvent(new Event('zizi-fonts-loaded'));
}

/**
 * Load inline font-face declarations from deferred style blocks
 */
function loadInlineFontFaces() {
    var deferredStyles = document.querySelectorAll('style[data-defer-fonts]');
    deferredStyles.forEach(function(style) {
        var css = style.getAttribute('data-defer-fonts');
        if (css) {
            style.textContent = css;
            style.removeAttribute('data-defer-fonts');
        }
    });
}

/**
 * Optimize WebFont loading with Font Loading API
 * Enhanced with better fallback handling
 */
function optimizeWebFontLoading() {
    // Check if Font Loading API is supported
    if (!('fonts' in document)) {
        // Fallback to timeout-based loading
        setTimeout(loadDeferredFonts, FONTS_CONFIG.maxFontLoadWait);
        return;
    }
    
    // Wait for fonts to load, then trigger deferred loading
    document.fonts.ready.then(function() {
        // Load any remaining deferred fonts
        loadDeferredFonts();
    }).catch(function(error) {
        console.warn('ZiziCache: Font loading error:', error);
        // Load deferred fonts anyway
        loadDeferredFonts();
    });
    
    // Set maximum wait time
    setTimeout(function() {
        if (!fontLoadingState.interactionTriggered) {
            loadDeferredFonts();
        }
    }, FONTS_CONFIG.maxFontLoadWait);
}

/**
 * Handle Google Fonts optimization
 * Enhanced with better URL parameter management
 */
function optimizeGoogleFonts() {
    var googleFontLinks = document.querySelectorAll('link[href*="fonts.googleapis.com/css"]');
    
    googleFontLinks.forEach(function(link) {
        var url = new URL(link.href);
        
        // Add font-display=swap if not present
        if (!url.searchParams.has('display')) {
            url.searchParams.set('display', FONTS_CONFIG.display);
            link.href = url.toString();
        }
        
        // Add preload for critical fonts
        if (!link.hasAttribute('data-defer')) {
            var preloadLink = document.createElement('link');
            preloadLink.rel = 'preload';
            preloadLink.as = 'style';
            preloadLink.href = link.href;
            preloadLink.onload = function() {
                this.onload = null;
                this.rel = 'stylesheet';
            };
            document.head.insertBefore(preloadLink, link);
        }
    });
}

/**
 * Handle Typekit/Adobe Fonts optimization
 */
function optimizeTypekitFonts() {
    var typekitLinks = document.querySelectorAll('link[href*="use.typekit.net"], script[src*="use.typekit.net"]');
    
    // Add preconnect for Typekit
    if (typekitLinks.length > 0) {
        addFontPreconnects();
        
        // Add font-display CSS for Typekit fonts
        var typekitCSS = document.createElement('style');
        typekitCSS.textContent = '.tk-loading { font-display: swap; }';
        document.head.appendChild(typekitCSS);
    }
}

/**
 * Initialize font optimization
 * Enhanced with better lifecycle handling
 */
function initializeFontOptimization() {
    if (fontLoadingState.initiated) return;
    fontLoadingState.initiated = true;
    
    // Load critical fonts immediately
    loadCriticalFonts();
    
    // Optimize Google Fonts
    optimizeGoogleFonts();
    
    // Optimize Typekit fonts
    optimizeTypekitFonts();
    
    // Set up deferred font loading
    setupDeferredFontLoading();
    
    // Use Font Loading API if available
    optimizeWebFontLoading();
}

/**
 * Set up deferred font loading triggers
 * Enhanced with multiple interaction types
 */
function setupDeferredFontLoading() {
    var interactionEvents = ['click', 'scroll', 'keydown', 'touchstart', 'mousemove'];
    var interactionTimeout;
    
    function triggerDeferredLoading() {
        loadDeferredFonts();
        
        // Clean up event listeners
        interactionEvents.forEach(function(event) {
            document.removeEventListener(event, triggerDeferredLoading, {passive: true});
        });
        
        clearTimeout(interactionTimeout);
    }
    
    // Listen for user interactions
    interactionEvents.forEach(function(event) {
        document.addEventListener(event, triggerDeferredLoading, {passive: true});
    });
    
    // Fallback timeout
    interactionTimeout = setTimeout(triggerDeferredLoading, 4000);
}

/**
 * Enhanced local font loading with better WOFF2 support
 */
function enhanceLocalFonts() {
    // Find font-face declarations in stylesheets
    var stylesheets = document.styleSheets;
    
    for (var i = 0; i < stylesheets.length; i++) {
        try {
            var rules = stylesheets[i].cssRules || stylesheets[i].rules;
            if (!rules) continue;
            
            for (var j = 0; j < rules.length; j++) {
                var rule = rules[j];
                if (rule.type === CSSRule.FONT_FACE_RULE) {
                    // Check if font-display is set
                    if (!rule.style.fontDisplay) {
                        rule.style.fontDisplay = FONTS_CONFIG.display;
                    }
                }
            }
        } catch (e) {
            // Cross-origin stylesheets might throw errors
            console.debug('ZiziCache: Could not access stylesheet rules:', e);
        }
    }
}

/**
 * Add resource hints for better font loading performance
 */
function addFontResourceHints() {
    // Add dns-prefetch for font providers
    var dnsPrefetchOrigins = [
        '//fonts.googleapis.com',
        '//fonts.gstatic.com',
        '//use.typekit.net',
        '//p.typekit.net'
    ];
    
    dnsPrefetchOrigins.forEach(function(origin) {
        var existingHint = document.querySelector('link[rel="dns-prefetch"][href="' + origin + '"]');
        if (!existingHint) {
            var dnsPrefetchLink = document.createElement('link');
            dnsPrefetchLink.rel = 'dns-prefetch';
            dnsPrefetchLink.href = origin;
            document.head.appendChild(dnsPrefetchLink);
        }
    });
}

// Initialize when DOM is ready
if (document.readyState === 'loading') {
    document.addEventListener('DOMContentLoaded', function() {
        initializeFontOptimization();
        enhanceLocalFonts();
        addFontResourceHints();
    });
} else {
    initializeFontOptimization();
    enhanceLocalFonts();
    addFontResourceHints();
}

// Handle dynamic font additions
var fontObserver = new MutationObserver(function(mutations) {
    var shouldReoptimize = false;
    
    mutations.forEach(function(mutation) {
        mutation.addedNodes.forEach(function(node) {
            if (node.nodeType === Node.ELEMENT_NODE) {
                // Check for new font links
                if (node.tagName === 'LINK' && node.href && 
                    (node.href.includes('fonts.googleapis.com') || node.href.includes('use.typekit.net'))) {
                    shouldReoptimize = true;
                }
                
                // Check for new font links in added subtrees
                var newFontLinks = node.querySelectorAll && 
                    node.querySelectorAll('link[href*="fonts.googleapis.com"], link[href*="use.typekit.net"]');
                if (newFontLinks && newFontLinks.length > 0) {
                    shouldReoptimize = true;
                }
            }
        });
    });
    
    if (shouldReoptimize) {
        setTimeout(function() {
            optimizeGoogleFonts();
            optimizeTypekitFonts();
        }, 100);
    }
});

// Start observing
fontObserver.observe(document.head, {
    childList: true,
    subtree: true
});

// Cleanup observer when page unloads
window.addEventListener('beforeunload', function() {
    fontObserver.disconnect();
});

})();
