/**
 * ZiziCache Enhanced Quicklink Configuration v2.0.0
 * Enhanced with better loading strategies and competitive improvements
 */
(function() {
'use strict';

// Enhanced configuration with better defaults
var quicklinkConfig = {
    // Core settings
    threshold: parseFloat(window.ziziQuicklinkConfig?.threshold) || 0.5,
    delay: parseInt(window.ziziQuicklinkConfig?.delay) || 0,
    timeout: parseInt(window.ziziQuicklinkConfig?.timeout) || 2000,
    priority: window.ziziQuicklinkConfig?.priority === 'true',
    
    // Enhanced performance settings
    concurrency: parseInt(window.ziziQuicklinkConfig?.concurrency) || 4,
    throttle: parseInt(window.ziziQuicklinkConfig?.throttle) || 250,
    
    // Network awareness settings
    respectReducedMotion: true,
    respectDataSaver: true,
    respectBattery: true,
    
    // URL patterns
    origins: window.ziziQuicklinkConfig?.origins || true,
    ignores: buildIgnorePatterns(),
    
    // Enhanced prefetch options
    el: document.body,
    
    // Callback functions for monitoring
    onError: handleQuicklinkError,
    onSuccess: handleQuicklinkSuccess
};

/**
 * Build ignore patterns from configuration
 * Enhanced with better pattern matching
 */
function buildIgnorePatterns() {
    var patterns = [];
    
    // Default ignore patterns
    var defaultIgnores = [
        // File downloads
        /\.(zip|rar|tar|gz|pdf|doc|docx|xls|xlsx|ppt|pptx)$/i,
        /\.(jpg|jpeg|png|gif|webp|avif|svg|ico)$/i,
        /\.(mp4|avi|mov|wmv|flv|webm|mp3|wav|ogg)$/i,
        
        // External links
        /^mailto:|^tel:|^sms:|^ftp:|^skype:/,
        /^https?:\/\/(?!.*yourdomain\.com)/,
        
        // WordPress specific
        /\/wp-admin|\/wp-login|\/wp-register/,
        /\?.*preview=|preview_id=|customize_changeset_uuid=/,
        /\/feed\/?$|\/rss\/?$|\/atom\/?$/,
        
        // eCommerce
        /\/cart|\/checkout|\/my-account|\/order/,
        /add-to-cart|remove_item|apply_coupon/,
        
        // Forms and actions
        /\?.*action=|&action=/,
        /\/contact|\/form/,
        
        // Enhanced ignore patterns from competitive analysis
        /logout|sign-out|disconnect/,
        /delete|remove|trash/,
        /\?.*utm_|&utm_/,
        /\#.*$/,
        /javascript:|data:/,
        
        // Page builders (prevent prefetching edit modes)
        /elementor-preview|et_fb=|builder=|ct_builder=|brizy-edit-iframe/,
        
        // Membership and authentication
        /register|signup|login|signin|password/
    ];
    
    patterns = patterns.concat(defaultIgnores);
    
    // Add custom ignore patterns from config
    if (window.ziziQuicklinkConfig?.customIgnores) {
        try {
            var customIgnores = JSON.parse(window.ziziQuicklinkConfig.customIgnores);
            if (Array.isArray(customIgnores)) {
                patterns = patterns.concat(customIgnores);
            }
        } catch (error) {
            console.warn('ZiziCache: Invalid custom ignore patterns:', error);
        }
    }
    
    return patterns;
}

/**
 * Enhanced network condition checking
 */
function checkNetworkConditions() {
    // Check for reduced motion preference
    if (window.matchMedia && window.matchMedia('(prefers-reduced-motion: reduce)').matches) {
        return false;
    }
    
    // Check for data saver mode
    if (navigator.connection && navigator.connection.saveData) {
        return false;
    }
    
    // Check for slow network
    if (navigator.connection) {
        var connection = navigator.connection;
        
        // Skip on very slow connections
        if (connection.effectiveType === 'slow-2g' || connection.effectiveType === '2g') {
            return false;
        }
        
        // Skip on low battery (if battery API available)
        if (navigator.getBattery) {
            navigator.getBattery().then(function(battery) {
                if (battery.level < 0.2 && !battery.charging) {
                    return false;
                }
            });
        }
    }
    
    return true;
}

/**
 * Enhanced error handling with logging
 */
function handleQuicklinkError(error) {
    console.warn('ZiziCache Quicklink Error:', error);
    
    // Send error data for analytics if configured
    if (window.ziziQuicklinkConfig?.analytics) {
        // Optional: Send error data to analytics
        if (window.gtag) {
            gtag('event', 'quicklink_error', {
                'error_message': error.message || 'Unknown error',
                'custom_parameter': 'zizi_cache'
            });
        }
    }
}

/**
 * Enhanced success handling with analytics
 */
function handleQuicklinkSuccess(url) {
    console.debug('ZiziCache Quicklink Success:', url);
    
    // Send success data for analytics if configured
    if (window.ziziQuicklinkConfig?.analytics) {
        if (window.gtag) {
            gtag('event', 'quicklink_prefetch', {
                'prefetched_url': url,
                'custom_parameter': 'zizi_cache'
            });
        }
    }
}

/**
 * Enhanced initialization with fallback support
 */
function initializeQuicklink() {
    // Check if quicklink library is loaded
    if (typeof quicklink === 'undefined') {
        console.warn('ZiziCache: Quicklink library not loaded');
        return;
    }
    
    // Check network conditions before initializing
    if (!checkNetworkConditions()) {
        console.info('ZiziCache: Quicklink disabled due to network conditions');
        return;
    }
    
    try {
        // Different initialization strategies based on loading method
        var strategy = window.ziziQuicklinkConfig?.strategy || 'intersection';
        
        switch (strategy) {
            case 'instant':
                // Initialize instant click prefetching
                quicklink.listen({
                    ...quicklinkConfig,
                    el: document.body,
                    trigger: 'mousedown' // Prefetch on mousedown for instant navigation
                });
                break;
                
            case 'hover':
                // Initialize hover prefetching with delay
                quicklink.listen({
                    ...quicklinkConfig,
                    el: document.body,
                    trigger: 'mouseenter',
                    delay: 100 // Small delay to prevent excessive prefetching
                });
                break;
                
            case 'intersection':
            default:
                // Default intersection observer strategy
                quicklink.listen(quicklinkConfig);
                break;
        }
        
        console.info('ZiziCache: Quicklink initialized with strategy:', strategy);
        
        // Optional: Add visual indicators for prefetched links
        if (window.ziziQuicklinkConfig?.visualIndicators) {
            addVisualIndicators();
        }
        
    } catch (error) {
        handleQuicklinkError(error);
    }
}

/**
 * Add visual indicators for prefetched links
 * Enhanced UX feature inspired by competitive analysis
 */
function addVisualIndicators() {
    var style = document.createElement('style');
    style.textContent = `
        .zizi-prefetched {
            position: relative;
        }
        .zizi-prefetched::after {
            content: '';
            position: absolute;
            bottom: -2px;
            left: 0;
            right: 0;
            height: 2px;
            background: linear-gradient(90deg, #0073aa, #005177);
            opacity: 0.3;
            border-radius: 1px;
        }
        .zizi-prefetching {
            opacity: 0.8;
        }
    `;
    document.head.appendChild(style);
    
    // Listen for prefetch events to add classes
    document.addEventListener('quicklink:prefetch', function(event) {
        var links = document.querySelectorAll('a[href="' + event.detail.url + '"]');
        links.forEach(function(link) {
            link.classList.add('zizi-prefetching');
        });
    });
    
    document.addEventListener('quicklink:prefetched', function(event) {
        var links = document.querySelectorAll('a[href="' + event.detail.url + '"]');
        links.forEach(function(link) {
            link.classList.remove('zizi-prefetching');
            link.classList.add('zizi-prefetched');
        });
    });
}

/**
 * Enhanced loading with better timing
 */
function enhancedInit() {
    // Wait for DOM and quicklink library
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', function() {
            setTimeout(initializeQuicklink, 100);
        });
    } else {
        setTimeout(initializeQuicklink, 100);
    }
}

// Initialize when ready
enhancedInit();

// Expose configuration for debugging
window.ziziQuicklinkConfigApplied = quicklinkConfig;

})();
