!(function (e) {
    "function" == typeof define && define.amd ? define(e) : e();
})(function () {
    // Konfigurace s možností deaktivace
    var config = window.ziziCacheLazyConfig || {
        rootMargin: "200px 0px",
        threshold: 0.01,
        debounceDelay: 50,
        debug: false,
        disabled: false  // Možnost úplně deaktivovat funkcionalitu
    };
    
    // Pokud je lazy loading deaktivován, okamžitě vykreslit všechny elementy
    if (config.disabled) {
        setTimeout(function() {
            document.querySelectorAll("div[data-zizi-cache-lazy-render]").forEach(function(el) {
                renderElement(el);
            });
        }, 0);
        return;
    }
    
    function logDebug(message) {
        if (config.debug && console && console.debug) {
            console.debug('[ZiziCache LazyRender]', message);
        }
    }
    
    // Bezpečnější forEach pro NodeList (kompatibilita se starými prohlížeči)
    function safeForEach(nodeList, callback) {
        if (nodeList.forEach) {
            nodeList.forEach(callback);
        } else {
            // Fallback pro IE
            for (var i = 0; i < nodeList.length; i++) {
                callback(nodeList[i], i, nodeList);
            }
        }
    }
    
    // Fallback pro prohlížeče bez IntersectionObserver
    function createFallbackLazyLoad() {
        logDebug('IntersectionObserver není podporován, používám fallback řešení');
        
        var activeElements = new WeakMap();  // Sledování aktivních elementů
        
        function checkVisibility() {
            var elements = document.querySelectorAll("div[data-zizi-cache-lazy-render]");
            var hasActiveElements = false;
            
            safeForEach(elements, function(element) {
                hasActiveElements = true;
                var rect = element.getBoundingClientRect();
                var isVisible = (
                    rect.top <= (window.innerHeight || document.documentElement.clientHeight) + 200 && 
                    rect.bottom >= -200
                );
                
                if (isVisible) {
                    renderElement(element);
                    // Odstranit atribut, abychom ho znovu neobsluhovali
                    element.removeAttribute('data-zizi-cache-lazy-render');
                    activeElements.delete(element);
                } else {
                    activeElements.set(element, true);
                }
            });
            
            // Pokud již neexistují žádné aktivní elementy, vyčistit event listenery
            if (!hasActiveElements) {
                cleanup();
            }
            
            return hasActiveElements;
        }
        
        // Vyčištění event listenerů když již nejsou potřeba
        function cleanup() {
            logDebug('Všechny lazy elementy zpracovány, čistím event listenery');
            window.removeEventListener('scroll', debouncedCheck);
            window.removeEventListener('resize', debouncedCheck);
            window.removeEventListener('orientationchange', debouncedCheck);
        }
        
        // Debounce funkce pro optimalizaci výkonu
        function debounce(func, delay) {
            var timeout;
            return function() {
                clearTimeout(timeout);
                timeout = setTimeout(func, delay);
            };
        }
        
        var debouncedCheck = debounce(checkVisibility, config.debounceDelay);
        
        window.addEventListener('scroll', debouncedCheck, { passive: true });
        window.addEventListener('resize', debouncedCheck, { passive: true });
        window.addEventListener('orientationchange', debouncedCheck);
        
        // Spustit počáteční kontrolu
        return checkVisibility();
    }
    
    // Funkce pro vykreslení obsahu z noscript tagu
    function renderElement(element) {
        try {
            var noscript = element.querySelector("noscript");
            if (noscript) {
                var parser = new DOMParser();
                var content = parser.parseFromString(noscript.textContent, "text/html");
                
                if (content.body.childNodes.length > 0) {
                    // Zajistit, že replaceWith je podporován, jinak použít jiný přístup
                    if (typeof element.replaceWith === 'function') {
                        element.replaceWith.apply(element, content.body.childNodes);
                    } else {
                        // Fallback pro starší prohlížeče
                        var parent = element.parentNode;
                        if (parent) {
                            var fragment = document.createDocumentFragment();
                            while (content.body.firstChild) {
                                fragment.appendChild(content.body.firstChild);
                            }
                            parent.replaceChild(fragment, element);
                        }
                    }
                    logDebug('Element úspěšně nahrazen obsahem');
                }
            } else {
                logDebug('Element neobsahuje noscript tag: ' + element.outerHTML.substring(0, 100));
            }
        } catch (err) {
            logDebug('Chyba při renderování elementu: ' + err.message);
        }
    }
    
    function init() {
        var elements = document.querySelectorAll("div[data-zizi-cache-lazy-render]");
        logDebug('Nalezeno ' + elements.length + ' lazy render elementů');
        
        if (elements.length) {
            if ('IntersectionObserver' in window) {
                var observer = new IntersectionObserver(
                    function (entries, obs) {
                        entries.forEach(function (entry) {
                            if (entry.isIntersecting) {
                                renderElement(entry.target);
                                obs.unobserve(entry.target);
                                
                                // Zkontrolovat, zda ještě existují elementy k observování
                                if (document.querySelectorAll("div[data-zizi-cache-lazy-render]").length === 0) {
                                    obs.disconnect();
                                    logDebug('Všechny elementy zpracovány, odpojuji observer');
                                }
                            }
                        });
                    },
                    { 
                        root: null, 
                        rootMargin: config.rootMargin,
                        threshold: config.threshold
                    }
                );
                
                safeForEach(elements, function(element) {
                    observer.observe(element);
                });
            } else {
                // Fallback pro starší prohlížeče
                createFallbackLazyLoad();
            }
        }
    }
    
    // Pokud je DOM již načtený, spustíme inicializaci ihned, jinak počkáme na DOMContentLoaded
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
});