!(function (e) {
    "function" == typeof define && define.amd ? define(e) : e();
})(function () {
    // Enhanced configuration with hybrid support
    var config = window.ziziCacheLazyConfig || {
        rootMargin: "200px 0px",
        threshold: 0.01,
        debounceDelay: 50,
        debug: false,
        disabled: false,
        method: 'hybrid',           // hybrid, css, javascript
        heightCache: true,          // Cache element heights
        fallbackDelay: 100         // Delay before fallback activation
    };
    
    // Content-visibility feature detection
    var supportsContentVisibility = (function() {
        if (typeof CSS === 'undefined' || !CSS.supports) return false;
        return CSS.supports('content-visibility', 'auto');
    })();
    
    // Pokud je lazy loading deaktivován, okamžitě vykreslit všechny elementy
    if (config.disabled) {
        setTimeout(function() {
            renderAllElements();
        }, 0);
        return;
    }
    
    function logDebug(message) {
        if (config.debug && console && console.debug) {
            console.debug('[ZiziCache Enhanced LazyRender]', message);
        }
    }
    
    // Render all lazy elements immediately (fallback/disabled mode)
    function renderAllElements() {
        var elements = document.querySelectorAll("div[data-zizi-cache-lazy-render]");
        safeForEach(elements, function(el) {
            renderElement(el);
        });
        
        var hybridElements = document.querySelectorAll(".zizi-lazy-hybrid[data-zizi-lazy-fallback]");
        safeForEach(hybridElements, function(el) {
            activateHybridElement(el);
        });
    }
    
    // Bezpečnější forEach pro NodeList (kompatibilita se starými prohlížeči)
    function safeForEach(nodeList, callback) {
        if (nodeList && nodeList.forEach) {
            nodeList.forEach(callback);
        } else if (nodeList) {
            // Fallback pro IE
            for (var i = 0; i < nodeList.length; i++) {
                callback(nodeList[i], i, nodeList);
            }
        }
    }
    
    // CSS-only lazy rendering activation
    function activateCSSLazyRender() {
        if (!supportsContentVisibility) {
            logDebug('Content-visibility not supported, falling back to JavaScript');
            return false;
        }
        
        logDebug('Using CSS content-visibility lazy rendering');
        
        // CSS lazy rendering is handled purely by CSS content-visibility
        // We only need to set up height caching if enabled
        if (config.heightCache) {
            setupHeightCaching();
        }
        
        return true;
    }
    
    // Height caching for layout stability
    function setupHeightCaching() {
        var elementsWithHeight = document.querySelectorAll('[data-zizi-height]');
        safeForEach(elementsWithHeight, function(element) {
            var cachedHeight = element.getAttribute('data-zizi-height');
            if (cachedHeight && !element.style.containIntrinsicSize) {
                element.style.containIntrinsicSize = 'auto ' + cachedHeight + 'px';
                logDebug('Applied cached height: ' + cachedHeight + 'px to element');
            }
        });
    }
    
    // Hybrid lazy rendering (CSS + JavaScript fallback)
    function activateHybridLazyRender() {
        if (supportsContentVisibility) {
            logDebug('Using CSS content-visibility with JavaScript monitoring');
            
            // CSS content-visibility handles the heavy lifting
            setupHeightCaching();
            
            // Monitor hybrid elements for fallback activation
            setupHybridFallbackMonitoring();
            
        } else {
            logDebug('Content-visibility not supported, using JavaScript fallback');
            
            // Convert hybrid elements to JavaScript lazy loading
            activateJavaScriptFallback();
        }
    }
    
    // Monitor hybrid elements and activate JavaScript fallback if needed
    function setupHybridFallbackMonitoring() {
        var hybridElements = document.querySelectorAll('.zizi-lazy-hybrid[data-zizi-lazy-fallback]');
        
        if (hybridElements.length === 0) return;
        
        // Use a delayed approach to allow CSS to work first
        setTimeout(function() {
            safeForEach(hybridElements, function(element) {
                // Check if element might need JavaScript assistance
                if (shouldActivateJavaScriptFallback(element)) {
                    activateHybridElement(element);
                }
            });
        }, config.fallbackDelay);
    }
    
    // Determine if JavaScript fallback should be activated for hybrid element
    function shouldActivateJavaScriptFallback(element) {
        // Check if element is in viewport but content-visibility isn't working properly
        var rect = element.getBoundingClientRect();
        var inViewport = (
            rect.top < (window.innerHeight || document.documentElement.clientHeight) &&
            rect.bottom > 0
        );
        
        // If element is in viewport but has very small height, CSS might not be working
        if (inViewport && rect.height < 10) {
            logDebug('Activating JavaScript fallback for hybrid element (potential CSS failure)');
            return true;
        }
        
        return false;
    }
    
    // Activate JavaScript fallback for content-visibility unsupported browsers
    function activateJavaScriptFallback() {
        var hybridElements = document.querySelectorAll('.zizi-lazy-hybrid[data-zizi-lazy-fallback]');
        
        safeForEach(hybridElements, function(element) {
            activateHybridElement(element);
        });
        
        // Also handle traditional lazy render elements
        initJavaScriptLazyRender();
    }
    
    // Activate hybrid element using JavaScript
    function activateHybridElement(element) {
        // Remove CSS content-visibility (not supported or not working)
        element.style.contentVisibility = '';
        element.style.containIntrinsicSize = '';
        
        // Convert to JavaScript lazy loading
        if ('IntersectionObserver' in window) {
            var rootMargin = element.getAttribute('data-root-margin') || config.rootMargin;
            
            var observer = new IntersectionObserver(function(entries) {
                entries.forEach(function(entry) {
                    if (entry.isIntersecting) {
                        // Element is visible, ensure it's fully rendered
                        entry.target.style.contentVisibility = 'visible';
                        observer.unobserve(entry.target);
                        logDebug('Hybrid element activated via JavaScript');
                    }
                });
            }, {
                root: null,
                rootMargin: rootMargin,
                threshold: config.threshold
            });
            
            observer.observe(element);
        } else {
            // Very old browsers - just make visible
            element.style.contentVisibility = 'visible';
        }
    }
    
    // Traditional JavaScript lazy rendering using IntersectionObserver
    function initJavaScriptLazyRender() {
        var elements = document.querySelectorAll("div[data-zizi-cache-lazy-render]");
        logDebug('Found ' + elements.length + ' JavaScript lazy render elements');
        
        if (elements.length === 0) return;
        
        if ('IntersectionObserver' in window) {
            var observer = new IntersectionObserver(
                function (entries, obs) {
                    entries.forEach(function (entry) {
                        if (entry.isIntersecting) {
                            renderElement(entry.target);
                            obs.unobserve(entry.target);
                            
                            // Check if all elements are processed
                            if (document.querySelectorAll("div[data-zizi-cache-lazy-render]").length === 0) {
                                obs.disconnect();
                                logDebug('All JavaScript lazy elements processed, disconnecting observer');
                            }
                        }
                    });
                },
                { 
                    root: null, 
                    rootMargin: config.rootMargin,
                    threshold: config.threshold
                }
            );
            
            safeForEach(elements, function(element) {
                // Override root margin if specified on element
                var elementRootMargin = element.getAttribute('data-root-margin');
                if (elementRootMargin && elementRootMargin !== config.rootMargin) {
                    // Create separate observer for this element
                    var customObserver = new IntersectionObserver(
                        function (entries, obs) {
                            entries.forEach(function (entry) {
                                if (entry.isIntersecting) {
                                    renderElement(entry.target);
                                    obs.unobserve(entry.target);
                                    obs.disconnect();
                                }
                            });
                        },
                        { 
                            root: null, 
                            rootMargin: elementRootMargin,
                            threshold: config.threshold
                        }
                    );
                    customObserver.observe(element);
                } else {
                    observer.observe(element);
                }
            });
        } else {
            // Fallback for browsers without IntersectionObserver
            createFallbackLazyLoad();
        }
    }
    
    // Fallback pro prohlížeče bez IntersectionObserver
    function createFallbackLazyLoad() {
        logDebug('IntersectionObserver není podporován, používám fallback řešení');
        
        var activeElements = new WeakMap();
        
        function checkVisibility() {
            var elements = document.querySelectorAll("div[data-zizi-cache-lazy-render]");
            var hasActiveElements = false;
            
            safeForEach(elements, function(element) {
                hasActiveElements = true;
                var rect = element.getBoundingClientRect();
                var rootMargin = element.getAttribute('data-root-margin') || config.rootMargin;
                var margin = parseInt(rootMargin.split('px')[0], 10) || 200;
                
                var isVisible = (
                    rect.top <= (window.innerHeight || document.documentElement.clientHeight) + margin && 
                    rect.bottom >= -margin
                );
                
                if (isVisible) {
                    renderElement(element);
                    element.removeAttribute('data-zizi-cache-lazy-render');
                    activeElements.delete(element);
                } else {
                    activeElements.set(element, true);
                }
            });
            
            if (!hasActiveElements) {
                cleanup();
            }
            
            return hasActiveElements;
        }
        
        function cleanup() {
            logDebug('Všechny lazy elementy zpracovány, čistím event listenery');
            window.removeEventListener('scroll', debouncedCheck);
            window.removeEventListener('resize', debouncedCheck);
            window.removeEventListener('orientationchange', debouncedCheck);
        }
        
        function debounce(func, delay) {
            var timeout;
            return function() {
                clearTimeout(timeout);
                timeout = setTimeout(func, delay);
            };
        }
        
        var debouncedCheck = debounce(checkVisibility, config.debounceDelay);
        
        window.addEventListener('scroll', debouncedCheck, { passive: true });
        window.addEventListener('resize', debouncedCheck, { passive: true });
        window.addEventListener('orientationchange', debouncedCheck);
        
        return checkVisibility();
    }
    
    // Funkce pro vykreslení obsahu z noscript tagu (JavaScript lazy render)
    function renderElement(element) {
        try {
            var noscript = element.querySelector("noscript");
            if (noscript) {
                var parser = new DOMParser();
                var content = parser.parseFromString(noscript.textContent, "text/html");
                
                if (content.body.childNodes.length > 0) {
                    if (typeof element.replaceWith === 'function') {
                        element.replaceWith.apply(element, content.body.childNodes);
                    } else {
                        // Fallback pro starší prohlížeče
                        var parent = element.parentNode;
                        if (parent) {
                            var fragment = document.createDocumentFragment();
                            while (content.body.firstChild) {
                                fragment.appendChild(content.body.firstChild);
                            }
                            parent.replaceChild(fragment, element);
                        }
                    }
                    logDebug('JavaScript lazy element úspěšně nahrazen obsahem');
                }
            } else {
                logDebug('Element neobsahuje noscript tag: ' + element.outerHTML.substring(0, 100));
            }
        } catch (err) {
            logDebug('Chyba při renderování elementu: ' + err.message);
        }
    }
    
    // Main initialization function
    function init() {
        logDebug('Initializing enhanced lazy render with method: ' + config.method);
        
        // Choose appropriate lazy rendering method
        switch (config.method) {
            case 'css':
                if (!activateCSSLazyRender()) {
                    // Fallback to JavaScript if CSS not supported
                    logDebug('CSS method failed, falling back to JavaScript');
                    initJavaScriptLazyRender();
                }
                break;
                
            case 'javascript':
                initJavaScriptLazyRender();
                break;
                
            case 'hybrid':
            default:
                activateHybridLazyRender();
                break;
        }
        
        logDebug('Enhanced lazy render initialization complete');
    }
    
    // Initialize when DOM is ready
    if (document.readyState === 'loading') {
        document.addEventListener('DOMContentLoaded', init);
    } else {
        init();
    }
});
