// ZiZi Cache Font Intelligence - Klientský detektor fontů
console.log('🔗 ZiZi Font Detector: Script loaded at', new Date().toLocaleTimeString());

// Okamžitá kontrola prostředí
console.log('🌍 Environment check:', {
  hasWindow: typeof window !== 'undefined',
  hasDocument: typeof document !== 'undefined',
  readyState: document.readyState,
  ziziConfig: typeof window.ziziCacheConfig !== 'undefined' ? window.ziziCacheConfig : 'NOT FOUND'
});

window.ZiziCache = window.ZiziCache || {};
ZiziCache.FontDetector = {
  config: {
    sampleRate: 0.03,  // 3% návštěv
    sessionKey: 'zizi-fonts-analyzed',
    aboveFoldHeight: 1200
  },  init: function() {
    console.log('🚀 ZiZi Font Detector: Initializing...');
    
    // Pro adminy se debug spustí vždy, jinak použij sampling
    const isAdmin = window.ziziCacheConfig && window.ziziCacheConfig.isAdmin;
    
    if (isAdmin) {
      console.log('👑 ZiZi Font Detector: Admin detected, running analysis');
    }
    
    // Sampling + kontrola session (přeskočit pro adminy)
    if (!isAdmin && (Math.random() > this.config.sampleRate || sessionStorage.getItem(this.config.sessionKey))) {
      console.log('⏭️ ZiZi Font Detector: Skipping due to sampling or session');
      return;
    }
    
    // Odložené spuštění s nízkou prioritou
    if ('requestIdleCallback' in window) {
      requestIdleCallback(() => this.analyze(), { timeout: 3000 });
    } else {
      setTimeout(() => this.analyze(), 3000);
    }
  },
    analyze: function() {
    try {
      // 1. Fonty načtené v prohlížeči
      const loadedFonts = this.getLoadedFonts();
      
      // 2. Fonty použité nad skládkou
      const aboveFoldFonts = this.getAboveFoldFonts();
      
      // 3. Debug výpis pro administrátory
      if (window.ziziCacheConfig && window.ziziCacheConfig.isAdmin) {
        this.debugLogForAdmin(loadedFonts, aboveFoldFonts);
      }
      
      this.sendResults({
        pageUrl: window.location.pathname,
        loadedFonts: loadedFonts,
        aboveFoldFonts: aboveFoldFonts
      });
    } catch (e) {
      // Izolované zachycení chyb
      console.debug('Font detection error:', e);
    }
  },
    getLoadedFonts: function() {
    if (!document.fonts || typeof document.fonts.values !== 'function') {
      return [];
    }
    
    const fontsMap = new Map();
    
    Array.from(document.fonts.values())
      .filter(font => font.status === "loaded")
      .forEach(font => {
        const key = font.family.replace(/['"]/g, '') + '-' + font.weight + '-' + font.style;
        if (!fontsMap.has(key)) {
          fontsMap.set(key, {
            family: font.family.replace(/['"]/g, ''),
            weight: font.weight,
            style: font.style
          });
        }
      });
    
    return Array.from(fontsMap.values());
  },
    getAboveFoldFonts: function() {
    // Zjednodušený selector
    const elements = document.querySelectorAll('h1, h2, h3, p, span, a, button');
    const fontsMap = new Map();
    
    Array.from(elements)
      .filter(el => {
        const rect = el.getBoundingClientRect();
        return rect.top < this.config.aboveFoldHeight && rect.top >= 0;
      })
      .forEach(el => {
        const style = window.getComputedStyle(el);
        const family = style.fontFamily.split(',')[0].trim().replace(/['"]/g, '');
        const key = family + '-' + style.fontWeight + '-' + style.fontStyle;
        
        if (!fontsMap.has(key)) {
          fontsMap.set(key, {
            family: family,
            weight: style.fontWeight,
            style: style.fontStyle,
            isHeading: /^h[1-3]$/i.test(el.tagName)
          });
        }
      });
      return Array.from(fontsMap.values());
  },
  
  debugLogForAdmin: function(loadedFonts, aboveFoldFonts) {
    // Podobný výpis jako váš test script
    console.log(
      "%c🔍 ZiZi Cache Font Intelligence Debug",
      "font-weight: bold; font-size: 1.4em; color: #0073aa; background: #f0f6fc; padding: 8px; border-radius: 4px;"
    );
    
    // Preloaded fonty
    const linkElements = document.querySelectorAll('link[rel="preload"]');
    const preloadedFonts = Array.from(linkElements).filter(link => link.as === "font");
    
    if (preloadedFonts.length > 0) {
      console.log(
        "%cFonts Preloaded via Resource Hints",
        "font-weight: bold; font-size: 1.2em; color: #d63384;"
      );
      preloadedFonts.forEach(font => console.log(`▸ ${font.href}`));
      console.log("");
    }
    
    // Načtené fonty
    if (loadedFonts.length > 0) {
      console.log(
        "%cFonts and Weights Loaded in the Document",
        "font-weight: bold; font-size: 1.2em; color: #d63384;"
      );
      loadedFonts.forEach(font => 
        console.log(`▸ ${font.family} - ${font.weight} - ${font.style}`)
      );
      console.log("");
    }
    
    // Above fold fonty
    if (aboveFoldFonts.length > 0) {
      console.log(
        "%cFonts and Weights Used Above the Fold",
        "font-weight: bold; font-size: 1.2em; color: #d63384;"
      );
      aboveFoldFonts.forEach(font => {
        const headingFlag = font.isHeading ? " 📍 HEADING" : "";
        console.log(`▸ ${font.family} | ${font.weight} | ${font.style}${headingFlag}`);
      });
      console.log("");
    }
    
    // Shrnutí pro optimalizaci
    const criticalFonts = aboveFoldFonts.filter(font => font.isHeading).slice(0, 2);
    if (criticalFonts.length > 0) {
      console.log(
        "%cCritical Fonts for Preload",
        "font-weight: bold; font-size: 1.2em; color: #198754;"
      );
      criticalFonts.forEach(font => 
        console.log(`🚀 ${font.family} - ${font.weight}`)
      );
    }
  },    sendResults: function(data) {
    sessionStorage.setItem(this.config.sessionKey, '1');
    
    // Prioritizovaný fetch s fallbackem na sendBeacon
    const payload = JSON.stringify(data);
    const url = '/wp-json/zizi-cache/v1/font-intelligence/collect';
    
    if (typeof fetch !== 'undefined') {
      fetch(url, {
        method: 'POST',
        headers: {
          'Content-Type': 'application/json',
        },
        body: payload
      }).catch(error => {
        console.debug('ZiZi Font Detector: Fetch failed, trying sendBeacon', error);
        if (navigator.sendBeacon) {
          navigator.sendBeacon(url, payload);
        }
      });
    } else if (navigator.sendBeacon) {
      navigator.sendBeacon(url, payload);
    }
  }
};

// Více robustní inicializace
console.log('📝 ZiZi Font Detector: Setting up initialization...');

function initializeFontDetector() {
  console.log('⚡ ZiZi Font Detector: Attempting to initialize...');
  console.log('📊 State check:', {
    readyState: document.readyState,
    hasZiziCache: typeof window.ZiziCache !== 'undefined',
    hasFontDetector: typeof window.ZiziCache?.FontDetector !== 'undefined',
    hasConfig: typeof window.ziziCacheConfig !== 'undefined'
  });
  
  if (typeof window.ZiziCache?.FontDetector?.init === 'function') {
    ZiziCache.FontDetector.init();
  } else {
    console.error('❌ ZiZi Font Detector: init function not available');
  }
}

// Múltiple fallback metódy inicializace
if (document.readyState === 'loading') {
  console.log('📄 Document still loading, waiting for DOMContentLoaded...');
  document.addEventListener('DOMContentLoaded', () => {
    console.log('✅ DOMContentLoaded fired');
    setTimeout(initializeFontDetector, 1000);
  });
} else {
  console.log('📄 Document already loaded, initializing immediately...');
  setTimeout(initializeFontDetector, 100);
}

// Fallback pro případ, že se config načte později
window.addEventListener('load', () => {
  console.log('🎯 Window load event fired');
  setTimeout(() => {
    if (typeof window.ziziCacheConfig !== 'undefined' && !sessionStorage.getItem('zizi-fonts-analyzed')) {
      console.log('🔄 Config found after load, trying init again...');
      initializeFontDetector();
    }
  }, 2000);
});
