/**
 * ZiziCache Core JavaScript Library v2.0.0
 * Enhanced with lifecycle events reliability and performance improvements
 * Inspired by Flying Press 5.0.6 improvements for lifecycle event reliability
 */
(function() {
'use strict';

// Store clicked elements during delay
var clickedElements = [];

// User interaction events that trigger script loading
var interactionEvents = ["click", "mousemove", "keydown", "touchstart", "touchmove", "wheel"];

// Find delayed scripts and stylesheets
var delayedScripts = document.querySelectorAll("script[data-src]");
var delayedStylesheets = document.querySelectorAll("link[data-href]");

// Lifecycle events tracking for reliable firing (inspired by Flying Press 5.0.6)
var lifecycleEventsFired = {
    DOMContentLoaded: false,
    load: false
};

// Original event listeners for restoration
var originalEventListeners = {
    document: {
        addEventListener: document.addEventListener.bind(document),
        removeEventListener: document.removeEventListener.bind(document)
    },
    window: {
        addEventListener: window.addEventListener.bind(window),
        removeEventListener: window.removeEventListener.bind(window)
    }
};

// Only proceed if there are delayed elements
if (delayedScripts.length || delayedStylesheets.length) {
    
    // Store click events during delay
    var storeClick = function(event) {
        clickedElements.push({
            target: event.target,
            view: event.view
        });
    };
    
    document.addEventListener("click", storeClick, {passive: true});
    
    // Set up interaction timeout
    var interactionTimeout = setTimeout(loadDelayedAssets, INTERACTION_TIMEOUT || 3000);
    
    // Listen for user interactions
    interactionEvents.forEach(function(eventType) {
        window.addEventListener(eventType, loadDelayedAssets, {passive: true});
    });
}

/**
 * Load all delayed assets and trigger lifecycle events
 */
function loadDelayedAssets() {
    // Clean up event listeners
    clearTimeout(interactionTimeout);
    interactionEvents.forEach(function(eventType) {
        window.removeEventListener(eventType, loadDelayedAssets, {passive: true});
    });
    
    // Preload scripts for better performance
    preloadDelayedScripts();
    
    // Load scripts sequentially to maintain dependency order
    if (delayedScripts.length) {
        loadScriptsSequentially(0);
    } else {
        // If no scripts, just trigger events and handle stylesheets
        triggerLifecycleEvents();
        loadDelayedStylesheets();
    }
    
    // Load delayed stylesheets
    loadDelayedStylesheets();
}

/**
 * Preload delayed scripts for performance (link rel=preload)
 */
function preloadDelayedScripts() {
    delayedScripts.forEach(function(script) {
        var src = script.getAttribute("data-src");
        if (src && !src.startsWith("data:")) {
            var preloadLink = document.createElement("link");
            preloadLink.rel = "preload";
            preloadLink.as = "script";
            preloadLink.href = src;
            document.head.appendChild(preloadLink);
        }
    });
}

/**
 * Load scripts sequentially to maintain dependency order
 * Enhanced with better lifecycle event handling (inspired by Flying Press 5.0.6)
 */
function loadScriptsSequentially(index) {
    if (index >= delayedScripts.length) {
        // All scripts loaded, trigger lifecycle events reliably
        triggerLifecycleEvents();
        return;
    }
    
    var script = delayedScripts[index];
    var newScript = document.createElement("script");
    
    // Copy all attributes except data-src
    Array.from(script.attributes).forEach(function(attr) {
        if (attr.name !== "data-src") {
            newScript.setAttribute(attr.name, attr.value);
        }
    });
    
    // Handle script loading
    var handleScriptLoad = function() {
        // Continue with next script
        loadScriptsSequentially(index + 1);
    };
    
    var handleScriptError = function(error) {
        console.warn("ZiziCache: Failed to load delayed script:", script.getAttribute("data-src"), error);
        // Continue with next script even on error
        loadScriptsSequentially(index + 1);
    };
    
    newScript.onload = handleScriptLoad;
    newScript.onerror = handleScriptError;
    
    // Set src to trigger loading
    var src = script.getAttribute("data-src");
    if (src) {
        newScript.src = src;
    } else {
        // Inline script
        newScript.textContent = script.textContent;
        // For inline scripts, trigger immediately
        setTimeout(handleScriptLoad, 0);
    }
    
    // Replace original script
    script.parentNode.replaceChild(newScript, script);
}

/**
 * Trigger lifecycle events reliably to ensure no duplicate triggers
 * Enhanced based on Flying Press 5.0.6 improvements
 */
function triggerLifecycleEvents() {
    // Remove click event storage
    document.removeEventListener("click", storeClick, {passive: true});
    
    // Trigger DOMContentLoaded if not already fired
    if (!lifecycleEventsFired.DOMContentLoaded) {
        lifecycleEventsFired.DOMContentLoaded = true;
        
        // Create and dispatch DOMContentLoaded event
        var domContentLoadedEvent = new Event("DOMContentLoaded", {
            bubbles: true,
            cancelable: false
        });
        document.dispatchEvent(domContentLoadedEvent);
        
        // Also trigger on window for compatibility
        window.dispatchEvent(domContentLoadedEvent);
    }
    
    // Trigger load event if not already fired
    if (!lifecycleEventsFired.load) {
        lifecycleEventsFired.load = true;
        
        var loadEvent = new Event("load", {
            bubbles: false,
            cancelable: false
        });
        window.dispatchEvent(loadEvent);
    }
    
    // Trigger custom ZiziCache event for other scripts to hook into
    var ziziLoadedEvent = new Event("zizi-cache-scripts-loaded", {
        bubbles: true,
        cancelable: false
    });
    document.dispatchEvent(ziziLoadedEvent);
    
    // Replay stored click events
    replayStoredClicks();
}

/**
 * Replay clicks that happened during delay
 */
function replayStoredClicks() {
    clickedElements.forEach(function(clickData) {
        try {
            var newEvent = new MouseEvent("click", {
                view: clickData.view,
                bubbles: true,
                cancelable: true
            });
            clickData.target.dispatchEvent(newEvent);
        } catch (error) {
            console.warn("ZiziCache: Failed to replay click event:", error);
        }
    });
    
    // Clear stored clicks
    clickedElements = [];
}

/**
 * Load delayed stylesheets
 */
function loadDelayedStylesheets() {
    delayedStylesheets.forEach(function(stylesheet) {
        var href = stylesheet.getAttribute("data-href");
        if (href) {
            stylesheet.href = href;
            stylesheet.removeAttribute("data-href");
        }
    });
}

/**
 * Load DCL (DOMContentLoaded) stylesheets
 */
function loadDCLStylesheets() {
    var dclStylesheets = document.querySelectorAll("link[data-dcl-href].zizi-dcl-css");
    if (dclStylesheets.length > 0) {
        var hrefs = [];
        dclStylesheets.forEach(function(stylesheet) {
            var href = stylesheet.getAttribute("data-dcl-href");
            if (href) {
                hrefs.push(href);
            }
        });
        
        hrefs.forEach(function(href) {
            var newStylesheet = document.createElement("link");
            newStylesheet.rel = "stylesheet";
            newStylesheet.type = "text/css";
            newStylesheet.href = href;
            document.head.appendChild(newStylesheet);
        });
    }
}

// Load DCL stylesheets when DOM is ready
if (document.readyState === "loading") {
    document.addEventListener("DOMContentLoaded", loadDCLStylesheets);
} else {
    loadDCLStylesheets();
}

/**
 * Iframe and video lazy loading with IntersectionObserver
 * Enhanced with better cleanup (inspired by Flying Press lazy-render improvements)
 */
var lazyObserver = new IntersectionObserver(function(entries) {
    entries.forEach(function(entry) {
        if (entry.isIntersecting) {
            // Stop observing this element
            lazyObserver.unobserve(entry.target);
            
            // Get lazy attributes and apply them
            var lazyAttributes = entry.target.getAttribute("data-lazy-attributes");
            if (lazyAttributes) {
                lazyAttributes.split(",").forEach(function(attribute) {
                    var lazyValue = entry.target.getAttribute("data-lazy-" + attribute);
                    if (lazyValue) {
                        entry.target.setAttribute(attribute, lazyValue);
                        // Clean up the data attribute
                        entry.target.removeAttribute("data-lazy-" + attribute);
                    }
                });
            }
            
            // Remove lazy loading data attributes for clean styling
            entry.target.removeAttribute("data-lazy-method");
            entry.target.removeAttribute("data-lazy-attributes");
            
            // Add loaded class for CSS targeting
            entry.target.classList.add("zizi-lazy-loaded");
        }
    });
}, {
    rootMargin: "300px",
    threshold: 0
});

// Observe all lazy elements
document.querySelectorAll('[data-lazy-method="viewport"]').forEach(function(element) {
    lazyObserver.observe(element);
});

// Cleanup when all elements are processed
function checkLazyCompletion() {
    var remainingElements = document.querySelectorAll('[data-lazy-method="viewport"]');
    if (remainingElements.length === 0) {
        lazyObserver.disconnect();
    }
}

// Check completion periodically
var completionCheckInterval = setInterval(function() {
    checkLazyCompletion();
    var remainingElements = document.querySelectorAll('[data-lazy-method="viewport"]');
    if (remainingElements.length === 0) {
        clearInterval(completionCheckInterval);
    }
}, 1000);

})();
