/**
 * ZiziCache Admin Interface
 * 
 * Toto je přepracované administrační rozhraní pro plugin ZiziCache.
 * Implementace používá čistý JavaScript bez závislosti na framework Vue.js.
 * Funkčnost původního admin rozhraní je plně zachována.
 */

// Zajistíme, že kód běží až po načtení DOMu
document.addEventListener('DOMContentLoaded', function() {
    // Reference na hlavní aplikační element
    const app = document.getElementById('app');
    
    // Získání konfigurace ze stránky
    const config = window.zizi_cache ? window.zizi_cache.config : (window.ziziCacheSettings ? window.ziziCacheSettings.config : {});
    const version = window.zizi_cache ? window.zizi_cache.version : (window.ziziCacheSettings ? window.ziziCacheSettings.version : '');
    const restUrl = window.zizi_cache ? window.zizi_cache.rest_url : (window.ziziCacheSettings ? window.ziziCacheSettings.restUrl : '');
    
    // Pokud není nastavena konfigurace, zobrazíme chybu
    if (!config || !restUrl) {
        app.innerHTML = '<div class="error">Chyba při načítání konfigurace.</div>';
        return;
    }
    
      // Základní stav aplikace
    const state = {
        activeTab: 'dashboard',
        config: typeof config === 'string' ? JSON.parse(config) : config,
        cachedPagesCount: 0,
        isLoading: false,
        notification: null,
        performanceMetrics: null,
        performanceLoading: false
    };

    // Inicializace konfigurace s novými vlastnostmi, pokud neexistují
    initializeConfig();
    
    // Inicializace aplikace
    initApp();
    
    /**
     * Inicializace konfigurace s novými vlastnostmi
     */
    function initializeConfig() {
        // Nastavení defaultních hodnot pro nové vlastnosti, pokud neexistují
        if (!state.config.cache_exclude_pages) {
            state.config.cache_exclude_pages = [];
        }
        
        if (!state.config.cache_ignore_queries) {
            state.config.cache_ignore_queries = [];
        }
        
        if (!state.config.cache_include_queries) {
            state.config.cache_include_queries = [];
        }
        
        if (!state.config.cache_bypass_cookies) {
            state.config.cache_bypass_cookies = [];
        }
        
        if (!state.config.host_third_party_locally) {
            state.config.host_third_party_locally = false;
        }
        
        if (!state.config.lazy_render_enabled) {
            state.config.lazy_render_enabled = false;
        }
        
        if (!state.config.lazy_render_selectors) {
            state.config.lazy_render_selectors = [];
        }
    }
      /**
     * Inicializace aplikace
     */
    function initApp() {
        // Načtení počtu cacheovaných stránek
        fetchCachedPagesCount();
        
        // Načtení performance metrik
        fetchPerformanceMetrics();
        
        // Vykreslení základní struktury aplikace
        renderApp();
        
        // Přidání posluchačů událostí
        addEventListeners();
    }
    
    /**
     * Vykreslení základní struktury aplikace
     */
    function renderApp() {
        app.innerHTML = `
            <div class="zizi-cache-admin">
                <header class="header bg-white border-b border-slate-200 p-4 md:p-5 flex items-center justify-between">
                    <div class="flex items-center">
                        <img src="${window.zizi_cache ? window.zizi_cache.plugin_url : ''}/assets/logo.svg" alt="ZiziCache" class="h-8 mr-2" />
                        <h1 class="text-xl font-semibold">ZiziCache</h1>
                        <span class="ml-2 text-xs bg-indigo-100 text-indigo-600 py-0.5 px-2 rounded">v${version}</span>
                    </div>
                </header>
                
                <div class="tab-navigation bg-white border-b border-slate-200 px-4 flex">
                    <button data-tab="dashboard" class="tab-button ${state.activeTab === 'dashboard' ? 'active' : ''}">
                        Dashboard
                    </button>
                    <button data-tab="caching" class="tab-button ${state.activeTab === 'caching' ? 'active' : ''}">
                        Caching
                    </button>
                    <button data-tab="optimization" class="tab-button ${state.activeTab === 'optimization' ? 'active' : ''}">
                        Optimization
                    </button>
                    <button data-tab="cdn" class="tab-button ${state.activeTab === 'cdn' ? 'active' : ''}">
                        CDN
                    </button>
                    <button data-tab="database" class="tab-button ${state.activeTab === 'database' ? 'active' : ''}">
                        Database
                    </button>
                    <button data-tab="advanced" class="tab-button ${state.activeTab === 'advanced' ? 'active' : ''}">
                        Advanced
                    </button>
                </div>
                
                <div id="notification-area" class="p-4 hidden"></div>
                
                <div id="tab-content" class="bg-slate-50 p-4 md:p-5">
                    ${renderTabContent()}
                </div>
            </div>
            
            <style>
                .zizi-cache-admin {
                    max-width: 1400px;
                    margin: 0 auto;
                    background: #fff;
                }
                .tab-button {
                    padding: 1rem;
                    font-weight: 500;
                    border-bottom: 2px solid transparent;
                    transition: all 0.2s;
                }
                .tab-button:hover {
                    color: #4F46E5;
                }
                .tab-button.active {
                    color: #4F46E5;
                    border-bottom-color: #4F46E5;
                }
                .card {
                    background: white;
                    border-radius: 0.5rem;
                    box-shadow: 0 1px 3px rgba(0, 0, 0, 0.1);
                    margin-bottom: 1rem;
                    overflow: hidden;
                }
                .card-header {
                    padding: 1rem 1.5rem;
                    border-bottom: 1px solid #f0f0f0;
                    font-weight: 600;
                    display: flex;
                    justify-content: space-between;
                    align-items: center;
                }
                .card-body {
                    padding: 1.5rem;
                }
                .form-group {
                    margin-bottom: 1.5rem;
                }
                .form-label {
                    display: block;
                    margin-bottom: 0.5rem;
                    font-weight: 500;
                }
                .form-input {
                    width: 100%;
                    padding: 0.5rem;
                    border: 1px solid #e2e8f0;
                    border-radius: 0.25rem;
                }
                .form-checkbox {
                    margin-right: 0.5rem;
                }
                .button {
                    padding: 0.5rem 1rem;
                    border-radius: 0.25rem;
                    font-weight: 500;
                    cursor: pointer;
                    transition: all 0.2s;
                }
                .button-primary {
                    background: #4F46E5;
                    color: white;
                    border: none;
                }
                .button-primary:hover {
                    background: #4338CA;
                }
                .button-secondary {
                    background: white;
                    color: #4F46E5;
                    border: 1px solid #4F46E5;
                }
                .button-secondary:hover {
                    background: #F9FAFB;
                }
                .spinner {
                    border: 3px solid rgba(0, 0, 0, 0.1);
                    border-radius: 50%;
                    border-top: 3px solid #4F46E5;
                    width: 20px;
                    height: 20px;
                    animation: spin 1s linear infinite;
                    display: inline-block;
                    vertical-align: middle;
                    margin-right: 0.5rem;
                }
                @keyframes spin {
                    0% { transform: rotate(0deg); }
                    100% { transform: rotate(360deg); }
                }
                .grid {
                    display: grid;
                    grid-template-columns: repeat(auto-fill, minmax(250px, 1fr));
                    gap: 1rem;
                }
                .notification {
                    padding: 1rem;
                    border-radius: 0.25rem;
                    margin-bottom: 1rem;
                }
                .notification-success {
                    background: #ECFDF5;
                    color: #047857;
                    border: 1px solid #A7F3D0;                }
                .notification-error {
                    background: #FEF2F2;
                    color: #B91C1C;
                    border: 1px solid #FECACA;
                }
                .notification-warning {
                    background: #FFFBEB;
                    color: #92400E;
                    border: 1px solid #FDE68A;
                }
                .performance-card {
                    transition: all 0.3s ease;
                    background: #fff;
                }
                .performance-card:hover {
                    box-shadow: 0 4px 12px rgba(0, 0, 0, 0.1);
                    transform: translateY(-2px);
                }
                .performance-card .performance-message {
                    transition: color 0.3s ease;
                }
                .performance-card details summary {
                    transition: all 0.2s ease;
                }
                .performance-card details summary:hover {
                    color: #4F46E5;
                }
                .performance-card details[open] summary {
                    margin-bottom: 8px;
                }
            </style>
        `;
    }
    
    /**
     * Vykreslení obsahu aktuální záložky
     */
    function renderTabContent() {
        switch (state.activeTab) {
            case 'dashboard':
                return renderDashboardTab();
            case 'caching':
                return renderCachingTab();
            case 'optimization':
                return renderOptimizationTab();
            case 'cdn':
                return renderCdnTab();
            case 'database':
                return renderDatabaseTab();
            case 'advanced':
                return renderAdvancedTab();
            default:
                return renderDashboardTab();
        }
    }
    
    /**
     * Vykreslení záložky Dashboard
     */
    function renderDashboardTab() {
        return `
            <div class="grid md:grid-cols-2 gap-4">
                <div class="card">
                    <div class="card-header">
                        Cache Status
                    </div>
                    <div class="card-body">
                        <div class="flex items-center justify-between mb-4">
                            <span>Cache enabled:</span>
                            <span class="px-2 py-1 bg-emerald-100 text-emerald-800 rounded">Yes</span>
                        </div>
                        <div class="flex items-center justify-between mb-4">
                            <span>Cached pages:</span>
                            <span id="cached-pages-count">${state.cachedPagesCount}</span>
                        </div>
                        <div class="flex items-center justify-between">
                            <span>Cache lifetime:</span>
                            <span>${state.config.cache_lifespan === 'never' ? 'Never expire' : state.config.cache_lifespan}</span>
                        </div>
                    </div>
                </div>
                
                <div class="card">
                    <div class="card-header">
                        Quick Actions
                    </div>
                    <div class="card-body">
                        <div class="grid grid-cols-1 md:grid-cols-2 gap-4">
                            <button id="purge-all-cache" class="button button-primary">
                                Purge All Cache
                            </button>
                            <button id="purge-and-preload" class="button button-secondary">
                                Purge and Preload
                            </button>
                            <button id="preload-cache" class="button button-secondary">
                                Preload Cache
                            </button>
                        </div>
                    </div>
                </div>
                
                <div class="card md:col-span-2">
                    <div class="card-header">
                        Optimization Status
                    </div>
                    <div class="card-body">
                        <div class="grid md:grid-cols-3 gap-4">
                            <div>
                                <h3 class="font-semibold mb-2">CSS Optimization</h3>
                                <div class="flex items-center justify-between mb-2">
                                    <span>Minify CSS:</span>
                                    <span class="${state.config.css_minify ? 'text-emerald-500' : 'text-slate-400'}">
                                        ${state.config.css_minify ? 'Enabled' : 'Disabled'}
                                    </span>
                                </div>
                                <div class="flex items-center justify-between">
                                    <span>Remove Unused CSS:</span>
                                    <span class="${state.config.css_rucss ? 'text-emerald-500' : 'text-slate-400'}">
                                        ${state.config.css_rucss ? 'Enabled' : 'Disabled'}
                                    </span>
                                </div>
                            </div>
                            
                            <div>
                                <h3 class="font-semibold mb-2">JavaScript Optimization</h3>
                                <div class="flex items-center justify-between mb-2">
                                    <span>Minify JS:</span>
                                    <span class="${state.config.js_minify ? 'text-emerald-500' : 'text-slate-400'}">
                                        ${state.config.js_minify ? 'Enabled' : 'Disabled'}
                                    </span>
                                </div>
                                <div class="flex items-center justify-between">
                                    <span>Defer JS:</span>
                                    <span class="${state.config.js_defer ? 'text-emerald-500' : 'text-slate-400'}">
                                        ${state.config.js_defer ? 'Enabled' : 'Disabled'}
                                    </span>
                                </div>
                            </div>
                            
                            <div>
                                <h3 class="font-semibold mb-2">Image Optimization</h3>
                                <div class="flex items-center justify-between mb-2">
                                    <span>Lazy Load Images:</span>
                                    <span class="${state.config.img_lazyload ? 'text-emerald-500' : 'text-slate-400'}">
                                        ${state.config.img_lazyload ? 'Enabled' : 'Disabled'}
                                    </span>
                                </div>                                <div class="flex items-center justify-between">
                                    <span>Responsive Images:</span>
                                    <span class="${state.config.img_responsive ? 'text-emerald-500' : 'text-slate-400'}">
                                        ${state.config.img_responsive ? 'Enabled' : 'Disabled'}
                                    </span>
                                </div>
                            </div>
                        </div>
                    </div>
                </div>
                  <!-- Performance Metrics Section -->
                <div class="card md:col-span-2">
                    <div class="card-header">
                        Výkon webu
                        ${state.performanceLoading ? '<span class="spinner ml-2"></span>' : ''}
                    </div>
                    <div class="card-body">
                        <div class="grid md:grid-cols-3 gap-4">
                            <!-- Database Performance Card -->
                            <div data-performance-card="database" class="performance-card p-4 border border-slate-200 rounded-lg">
                                <div class="performance-card-header" style="display: flex; align-items: center; margin-bottom: 12px;">
                                    <span class="performance-icon" style="font-size: 24px; margin-right: 8px;">📊</span>
                                    <div>
                                        <h4 class="performance-message" style="margin: 0; color: #666; font-weight: 600;">Načítání...</h4>
                                        <span class="performance-status" style="font-size: 12px; text-transform: uppercase; opacity: 0.7;">Databázový výkon</span>
                                    </div>
                                </div>
                                <p class="performance-description" style="margin: 0 0 8px 0; color: #666; font-size: 14px;">Kontroluji databázový výkon...</p>
                                <div class="performance-recommendation" style="font-size: 13px; color: #f59e0b; font-weight: 500;"></div>
                            </div>
                            
                            <!-- HTTP Performance Card -->
                            <div data-performance-card="http" class="performance-card p-4 border border-slate-200 rounded-lg">
                                <div class="performance-card-header" style="display: flex; align-items: center; margin-bottom: 12px;">
                                    <span class="performance-icon" style="font-size: 24px; margin-right: 8px;">🌐</span>
                                    <div>
                                        <h4 class="performance-message" style="margin: 0; color: #666; font-weight: 600;">Načítání...</h4>
                                        <span class="performance-status" style="font-size: 12px; text-transform: uppercase; opacity: 0.7;">Síťové připojení</span>
                                    </div>
                                </div>
                                <p class="performance-description" style="margin: 0 0 8px 0; color: #666; font-size: 14px;">Kontroluji síťové připojení...</p>
                                <div class="performance-recommendation" style="font-size: 13px; color: #f59e0b; font-weight: 500;"></div>
                            </div>
                            
                            <!-- Cache Performance Card -->
                            <div data-performance-card="cache" class="performance-card p-4 border border-slate-200 rounded-lg">
                                <div class="performance-card-header" style="display: flex; align-items: center; margin-bottom: 12px;">
                                    <span class="performance-icon" style="font-size: 24px; margin-right: 8px;">⚡</span>
                                    <div>
                                        <h4 class="performance-message" style="margin: 0; color: #666; font-weight: 600;">Načítání...</h4>
                                        <span class="performance-status" style="font-size: 12px; text-transform: uppercase; opacity: 0.7;">Cache systém</span>
                                    </div>
                                </div>
                                <p class="performance-description" style="margin: 0 0 8px 0; color: #666; font-size: 14px;">Kontroluji cache systém...</p>
                                <div class="performance-recommendation" style="font-size: 13px; color: #f59e0b; font-weight: 500;"></div>
                            </div>
                        </div>
                        
                        <div class="mt-6 pt-4 border-t border-slate-200">
                            <button id="refresh-performance" class="button button-secondary text-sm mr-2">
                                Obnovit metriky
                            </button>
                            <button id="test-http-performance" class="button button-outline text-sm">
                                Test síťového připojení
                            </button>
                        </div>
                    </div>
                </div>
            </div>
        `;
    }
    
    /**
     * Vykreslení záložky Caching
     */
    function renderCachingTab() {
        return `
            <div class="card">
                <div class="card-header">
                    Cache Settings
                </div>
                <div class="card-body">
                    <form id="caching-form">
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="cache_logged_in" ${state.config.cache_logged_in ? 'checked' : ''}>
                                Cache for logged-in users
                            </label>
                            <p class="text-sm text-slate-500">Enable caching for logged-in users. Not recommended for sites with user-specific content.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="cache_mobile" ${state.config.cache_mobile ? 'checked' : ''}>
                                Cache for mobile devices
                            </label>
                            <p class="text-sm text-slate-500">Create separate cache for mobile devices.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="cache_lifespan">Cache lifetime</label>
                            <select id="cache_lifespan" name="cache_lifespan" class="form-input">
                                <option value="never" ${state.config.cache_lifespan === 'never' ? 'selected' : ''}>Never expire</option>
                                <option value="1hour" ${state.config.cache_lifespan === '1hour' ? 'selected' : ''}>1 hour</option>
                                <option value="3hours" ${state.config.cache_lifespan === '3hours' ? 'selected' : ''}>3 hours</option>
                                <option value="6hours" ${state.config.cache_lifespan === '6hours' ? 'selected' : ''}>6 hours</option>
                                <option value="12hours" ${state.config.cache_lifespan === '12hours' ? 'selected' : ''}>12 hours</option>
                                <option value="1day" ${state.config.cache_lifespan === '1day' ? 'selected' : ''}>1 day</option>
                                <option value="3days" ${state.config.cache_lifespan === '3days' ? 'selected' : ''}>3 days</option>
                                <option value="1week" ${state.config.cache_lifespan === '1week' ? 'selected' : ''}>1 week</option>
                                <option value="2weeks" ${state.config.cache_lifespan === '2weeks' ? 'selected' : ''}>2 weeks</option>
                                <option value="1month" ${state.config.cache_lifespan === '1month' ? 'selected' : ''}>1 month</option>
                            </select>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="cache_preload" ${state.config.cache_preload ? 'checked' : ''}>
                                Preload cache
                            </label>
                            <p class="text-sm text-slate-500">Automatically preload cache after clearing.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="cache_exclude_pages">Exclude pages from cache</label>
                            <textarea id="cache_exclude_pages" name="cache_exclude_pages" class="form-input" rows="3" placeholder="One URL per line">${state.config.cache_exclude_pages.join('\n')}</textarea>
                            <p class="text-sm text-slate-500">Enter URLs to exclude from caching, one per line. Use * for wildcards.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="cache_ignore_queries">Ignore query parameters</label>
                            <textarea id="cache_ignore_queries" name="cache_ignore_queries" class="form-input" rows="3" placeholder="One parameter per line">${state.config.cache_ignore_queries.join('\n')}</textarea>
                            <p class="text-sm text-slate-500">Enter query parameters to ignore when generating cache, one per line.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="cache_include_queries">Include query parameters</label>
                            <textarea id="cache_include_queries" name="cache_include_queries" class="form-input" rows="3" placeholder="One parameter per line">${state.config.cache_include_queries.join('\n')}</textarea>
                            <p class="text-sm text-slate-500">Enter query parameters for which separate cache should be generated, one per line.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="cache_bypass_cookies">Bypass cookies</label>
                            <textarea id="cache_bypass_cookies" name="cache_bypass_cookies" class="form-input" rows="3" placeholder="One cookie per line">${state.config.cache_bypass_cookies.join('\n')}</textarea>
                            <p class="text-sm text-slate-500">Enter cookies that should bypass the cache, one per line.</p>
                        </div>
                        
                        <button type="submit" class="button button-primary">Save Changes</button>
                    </form>
                </div>
            </div>
        `;
    }
    
    /**
     * Vykreslení záložky Optimization
     */
    function renderOptimizationTab() {
        return `
            <div class="card">
                <div class="card-header">
                    CSS Optimization
                </div>
                <div class="card-body">
                    <form id="css-optimization-form">
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="css_minify" ${state.config.css_minify ? 'checked' : ''}>
                                Minify CSS
                            </label>
                            <p class="text-sm text-slate-500">Reduce the size of your CSS files by removing unnecessary characters.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="css_rucss" ${state.config.css_rucss ? 'checked' : ''}>
                                Remove Unused CSS
                            </label>
                            <p class="text-sm text-slate-500">Identify and remove CSS rules that aren't used on your pages.</p>
                        </div>
                        <div class="form-group css-rucss-options" style="${state.config.css_rucss ? '' : 'display:none;'}">
    <label class="form-label">Load unused CSS</label>
    <div class="mb-2">
        <label class="inline-flex items-center mb-2">
            <input type="radio" class="zizi-radio mr-2" name="css_rucss_method" value="async" ${state.config.css_rucss_method === 'async' ? 'checked' : ''}>
            Asynchronously
            <span class="text-sm text-slate-500 ml-2">Load unused CSS asynchronously after page load</span>
        </label>
    </div>
    <div class="mb-2">
        <label class="inline-flex items-center mb-2">
            <input type="radio" class="form-radio mr-2" name="css_rucss_method" value="interaction" ${state.config.css_rucss_method === 'interaction' ? 'checked' : ''}>
            On user interaction
            <span class="text-sm text-slate-500 ml-2">Load unused CSS on user interaction (mouse movement, scroll or keypress)</span>
        </label>
    </div>
    <div class="mb-2">
        <label class="inline-flex items-center mb-2">
            <input type="radio" class="form-radio mr-2" name="css_rucss_method" value="domcontentloaded" ${state.config.css_rucss_method === 'domcontentloaded' ? 'checked' : ''}>
            After DOMContentLoaded
            <span class="text-sm text-slate-500 ml-2">Load unused CSS after DOM is fully parsed (recommended)</span>
        </label>
    </div>
    <div class="mb-2">
        <label class="inline-flex items-center mb-2">
            <input type="radio" class="form-radio mr-2" name="css_rucss_method" value="remove" ${state.config.css_rucss_method === 'remove' ? 'checked' : ''}>
            Remove
            <span class="text-sm text-slate-500 ml-2">Completely remove unused CSS (not recommended for dynamic content)</span>
        </label>
    </div>
</div>

<div class="form-group css-rucss-options" style="${state.config.css_rucss ? '' : 'display:none;'}">
    <label class="form-label" for="css_rucss_exclude_stylesheets">Exclude stylesheets</label>
    <textarea id="css_rucss_exclude_stylesheets" name="css_rucss_exclude_stylesheets" class="form-input" rows="3" placeholder="Enter the URL or path of CSS files that has to be excluded from removing unused CSS">${state.config.css_rucss_exclude_stylesheets.join('\n')}</textarea>
    <p class="text-sm text-slate-500">Enter URLs or paths of CSS files to exclude from RUCSS processing, one per line.</p>
</div>

<div class="form-group css-rucss-options" style="${state.config.css_rucss ? '' : 'display:none;'}">
    <label class="form-label" for="css_rucss_include_selectors">Include selectors</label>
    <textarea id="css_rucss_include_selectors" name="css_rucss_include_selectors" class="form-input" rows="3" placeholder="Enter the list of CSS selectors that has to be included in used CSS">${state.config.css_rucss_include_selectors.join('\n')}</textarea>
    <p class="text-sm text-slate-500">Enter CSS selectors to always include, even if not detected in HTML, one per line.</p>
</div>
                        
                        <button type="submit" class="button button-primary">Save CSS Settings</button>
                    </form>
                </div>
            </div>
            
            <div class="card mt-4">
                <div class="card-header">
                    JavaScript Optimization
                </div>
                <div class="card-body">
                    <form id="js-optimization-form">
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="js_minify" ${state.config.js_minify ? 'checked' : ''}>
                                Minify JavaScript
                            </label>
                            <p class="text-sm text-slate-500">Reduce the size of your JavaScript files by removing unnecessary characters.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="js_defer" ${state.config.js_defer ? 'checked' : ''}>
                                Defer JavaScript
                            </label>
                            <p class="text-sm text-slate-500">Delay JavaScript execution until after the page is loaded, improving initial render times.</p>
                        </div>
<div class="form-group js-defer-options" style="${state.config.js_defer ? '' : 'display:none;'}">
    <label class="form-label" for="js_defer_excludes">Exclude scripts from defer</label>
    <textarea id="js_defer_excludes" name="js_defer_excludes" class="form-input" rows="3" placeholder="Enter unique keywords separated by new line">${state.config.js_defer_excludes ? state.config.js_defer_excludes.join('\n') : ''}</textarea>
    <p class="text-sm text-slate-500">Scripts that should be excluded from deferring. Enter unique keywords separated by new line.</p>
</div>                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="js_delay" ${state.config.js_delay ? 'checked' : ''}>
                                Delay JavaScript Execution
                            </label>
                            <p class="text-sm text-slate-500">Execute JS only after user interaction, significantly improving page loading times.</p>
                        </div>
<div class="form-group js-delay-options" style="${state.config.js_delay ? '' : 'display:none;'}">
    <label class="form-label">Delay method</label>
    <div class="mb-2">
        <label class="inline-flex items-center mb-2">
            <input type="radio" class="form-radio mr-2" name="js_delay_method" value="selected" ${state.config.js_delay_method === 'selected' ? 'checked' : ''}>
            Delay selected scripts
            <span class="text-sm text-slate-500 ml-2">Delay only the scripts that match the keywords below</span>
        </label>
    </div>
    <div class="mb-2">
        <label class="inline-flex items-center mb-2">
            <input type="radio" class="form-radio mr-2" name="js_delay_method" value="all" ${state.config.js_delay_method === 'all' ? 'checked' : ''}>
            Delay all scripts
            <span class="text-sm text-slate-500 ml-2">Delay all scripts except those that match the exclude keywords</span>
        </label>
    </div>
</div>

<div class="form-group js-delay-selected-options" style="${state.config.js_delay && state.config.js_delay_method === 'selected' ? '' : 'display:none;'}">
    <label class="form-label" for="js_delay_selected">Scripts to delay</label>
    <textarea id="js_delay_selected" name="js_delay_selected" class="form-input" rows="5" placeholder="Enter unique keywords separated by new line">${state.config.js_delay_selected.join('\n')}</textarea>
    <p class="text-sm text-slate-500">Enter keywords for scripts that should be delayed, one per line.</p>
</div>

<div class="form-group js-delay-all-options" style="${state.config.js_delay && state.config.js_delay_method === 'all' ? '' : 'display:none;'}">
    <label class="form-label" for="js_delay_all_excludes">Scripts to exclude from delay</label>
    <textarea id="js_delay_all_excludes" name="js_delay_all_excludes" class="form-input" rows="5" placeholder="Enter unique keywords separated by new line">${state.config.js_delay_all_excludes ? state.config.js_delay_all_excludes.join('\n') : ''}</textarea>
    <p class="text-sm text-slate-500">Enter keywords for scripts that should NOT be delayed, one per line.</p>
</div>

                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="host_third_party_locally" ${state.config.host_third_party_locally ? 'checked' : ''}>
                                Host Third Party CSS and JS Locally
                            </label>
                            <p class="text-sm text-slate-500">Download and serve external CSS and JS files from your server to improve performance and privacy.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="lazy_render_enabled" ${state.config.lazy_render_enabled ? 'checked' : ''}>
                                Lazy Render Elements
                            </label>
                            <p class="text-sm text-slate-500">Delay rendering of specified elements until they are visible in the viewport.</p>
                        </div>
                        
<div class="form-group">
    <label class="form-label" for="lazy_render_selectors">Lazy Render Selectors</label>
    <textarea id="lazy_render_selectors" name="lazy_render_selectors" class="form-input" rows="3" placeholder="One CSS selector per line">${state.config.js_lazy_render_selectors.join('\n')}</textarea>
    <p class="text-sm text-slate-500">Enter CSS selectors for elements to lazy render, one per line (e.g. .sidebar, #comments, footer).</p>
</div>
                        
                        <button type="submit" class="button button-primary">Save JavaScript Settings</button>
                    </form>
                </div>
            </div>
            
            <div class="card mt-4">
                <div class="card-header">
                    Image Optimization
                </div>
                <div class="card-body">
                    <form id="img-optimization-form">
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="img_lazyload" ${state.config.img_lazyload ? 'checked' : ''}>
                                Lazy Load Images
                            </label>
                            <p class="text-sm text-slate-500">Load images only when they appear in the viewport, significantly improving page load times.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="img_responsive" ${state.config.img_responsive ? 'checked' : ''}>
                                Responsive Images
                            </label>
                            <p class="text-sm text-slate-500">Automatically generate and serve appropriately sized images for different devices.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="img_width_height" ${state.config.img_width_height ? 'checked' : ''}>
                                Add Missing Width/Height
                            </label>
                            <p class="text-sm text-slate-500">Add width and height attributes to images to prevent layout shifts during page load.</p>
                        </div>
                        
                        <button type="submit" class="button button-primary">Save Image Settings</button>
                    </form>
                </div>
            </div>
              <div class="card mt-4">
                <div class="card-header">
                    Font Optimization
                </div>
                <div class="card-body">
                    <form id="font-optimization-form">
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="fonts_optimize_google_fonts" ${state.config.fonts_optimize_google_fonts ? 'checked' : ''}>
                                Optimize Google Fonts
                            </label>
                            <p class="text-sm text-slate-500">Optimize how Google Fonts are loaded on your site.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="fonts_display_swap" ${state.config.fonts_display_swap ? 'checked' : ''}>
                                Add Font Display Swap
                            </label>
                            <p class="text-sm text-slate-500">Show text using a fallback font while custom fonts are loading.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="fonts_allow_cdn_domains" ${state.config.fonts_allow_cdn_domains !== false ? 'checked' : ''}>
                                Allow CDN Domains for Fonts
                            </label>
                            <p class="text-sm text-slate-500">Allow fetching font resources from configured CDN domains.</p>
                        </div>
                        
                        <div class="form-group" style="margin-bottom: 2rem;">
                            <label class="form-label" for="fonts_atf_height">Above-the-fold Height (pixels)</label>
                            <input type="number" id="fonts_atf_height" name="fonts_atf_height" class="form-input" min="100" max="10000" value="${state.config.fonts_atf_height || 1200}">
                            <p class="text-sm text-slate-500">Height for above-the-fold content detection in pixels. Used for prioritizing visible fonts.</p>
                        </div>
                        
                        <div class="form-group" style="margin-bottom: 2rem;">
                            <label class="form-label" for="fonts_cache_ttl">Font Cache TTL (seconds)</label>
                            <input type="number" id="fonts_cache_ttl" name="fonts_cache_ttl" class="form-input" min="60" max="2592000" value="${state.config.fonts_cache_ttl || 3600}">
                            <p class="text-sm text-slate-500">Cache time-to-live for font resources in seconds. Higher values reduce server load but might delay font updates.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="fonts_debug" ${state.config.fonts_debug ? 'checked' : ''}>
                                Enable Debug Logging
                            </label>
                            <p class="text-sm text-slate-500">Enable detailed font optimization logging for troubleshooting (increases server load).</p>
                        </div>
                        
                        <button type="submit" class="button button-primary">Save Font Settings</button>
                    </form>
                </div>
            </div>
        `;
    }
    
    
    /**
     * Vykreslení záložky Database
     */
    function renderDatabaseTab() {
   
            return `
            <div class="card">
                <div class="card-header">
                    Database Cleanup
                </div>
                <div class="card-body">
                    <form id="database-form">
                        <p class="mb-4">Select what you want to clean up from your database:</p>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="db_post_revisions" ${state.config.db_post_revisions ? 'checked' : ''}>
                                Post Revisions
                            </label>
                            <p class="text-sm text-slate-500">Delete all post revisions from the database.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="db_post_auto_drafts" ${state.config.db_post_auto_drafts ? 'checked' : ''}>
                                Auto-Drafts
                            </label>
                            <p class="text-sm text-slate-500">Delete all auto-draft posts from the database.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="db_post_trashed" ${state.config.db_post_trashed ? 'checked' : ''}>
                                Trashed Posts
                            </label>
                            <p class="text-sm text-slate-500">Delete all posts from trash permanently.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="db_comments_spam" ${state.config.db_comments_spam ? 'checked' : ''}>
                                Spam Comments
                            </label>
                            <p class="text-sm text-slate-500">Delete all spam comments from the database.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="db_comments_trashed" ${state.config.db_comments_trashed ? 'checked' : ''}>
                                Trashed Comments
                            </label>
                            <p class="text-sm text-slate-500">Delete all comments from trash permanently.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="db_transients_expired" ${state.config.db_transients_expired ? 'checked' : ''}>
                                Expired Transients
                            </label>
                            <p class="text-sm text-slate-500">Delete all expired transient options from the database.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="db_transients_all" ${state.config.db_transients_all ? 'checked' : ''}>
                                All Transients
                            </label>
                            <p class="text-sm text-slate-500">Delete all transient options from the database (expired and non-expired).</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="db_optimize_tables" ${state.config.db_optimize_tables ? 'checked' : ''}>
                                Optimize Database Tables
                            </label>
                            <p class="text-sm text-slate-500">Run optimization queries on database tables to recover space.</p>
                        </div>

                        <div class="form-group">
                            <label class="form-label">
                                <input type="checkbox" class="form-checkbox" name="db_actionscheduler_clean" ${state.config.db_actionscheduler_clean ? 'checked' : ''}>
                                Action Scheduler Cleanup
                            </label>
                            <p class="text-sm text-slate-500">Pravidelně čistit staré záznamy z Action Scheduler tabulek podle stejného plánu jako ostatní databázové čištění.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="db_actionscheduler_logs_days">Odstranit záznamy z Action Scheduler logs starší než (dny):</label>
                            <input type="number" class="form-input" name="db_actionscheduler_logs_days" id="db_actionscheduler_logs_days"
                                   value="${state.config.db_actionscheduler_logs_days || 1}" min="1">
                            <p class="text-sm text-slate-500">Počet dní, po kterých budou záznamy v tabulce actionscheduler_logs odstraněny.</p>
                        </div>
                        
                        <div class="form-group">
                            <label class="form-label" for="db_actionscheduler_actions_days">Odstranit dokončené Action Scheduler akce starší než (dny):</label>
                            <input type="number" class="form-input" name="db_actionscheduler_actions_days" id="db_actionscheduler_actions_days"
                                   value="${state.config.db_actionscheduler_actions_days || 7}" min="1">
                            <p class="text-sm text-slate-500">Počet dní, po kterých budou dokončené, zrušené a neúspěšné akce v tabulce actionscheduler_actions odstraněny.</p>
                        </div>

                        <div class="form-group">
                            <label class="form-label" for="db_schedule_clean">Schedule Database Cleanup</label>
                            <select id="db_schedule_clean" name="db_schedule_clean" class="form-input">
                                <option value="never" ${state.config.db_schedule_clean === 'never' ? 'selected' : ''}>Never (Manual Only)</option>
                                <option value="daily" ${state.config.db_schedule_clean === 'daily' ? 'selected' : ''}>Daily</option>
                                <option value="weekly" ${state.config.db_schedule_clean === 'weekly' ? 'selected' : ''}>Weekly</option>
                                <option value="monthly" ${state.config.db_schedule_clean === 'monthly' ? 'selected' : ''}>Monthly</option>
                            </select>
                        </div>
                        
                        <div class="mt-4">
                            <button type="submit" class="button button-primary">Save Database Settings</button>
                            <button type="button" id="run-db-cleanup" class="button button-secondary ml-2">Run Cleanup Now</button>
                        </div>
                    </form>
                </div>
                <div class="card mt-4">
                    <div class="card-header">
                        ActionScheduler Status
                    </div>
                    <div class="card-body">
                        <div id="actionscheduler-status-loading" class="text-center p-4">
                            <div class="spinner" style="display: inline-block;"></div>
                            <p>Načítání informací o ActionScheduler...</p>
                        </div>
                        <div id="actionscheduler-status-content" class="hidden">
                            <div class="mb-4">
                                <h4 class="font-semibold mb-2">Statistiky tabulek</h4>
                                <p>Velikost tabulky <b>actionscheduler_actions</b>: <span id="as-actions-size">-</span> MB</p>
                                <p>Velikost tabulky <b>actionscheduler_logs</b>: <span id="as-logs-size">-</span> MB</p>
                                <p>Počet záznamů v <b>actionscheduler_logs</b>: <span id="as-logs-count">-</span></p>
                            </div>
                            
                            <div class="mb-4">
                                <h4 class="font-semibold mb-2">Počet akcí podle stavu</h4>
                                <p>Stav 'complete': <span id="as-complete-count">-</span></p>
                                <p>Stav 'pending': <span id="as-pending-count">-</span></p>
                                <p>Stav 'canceled': <span id="as-canceled-count">-</span></p>
                                <p>Stav 'failed': <span id="as-failed-count">-</span></p>
                            </div>
                            
                            <div class="mt-4">
                                <p class="text-sm text-slate-600">
                                    <b>Poznámka:</b> Čištění Action Scheduler proběhne automaticky podle nastaveného intervalu v sekci "Schedule Database Cleanup" nebo pomocí tlačítka "Run Cleanup Now" výše.
                                    Čištění odstraní z tabulky logs záznamy starší než nastavený počet dní a z tabulky actions všechny záznamy se stavem 'complete', 'canceled' a 'failed',
                                    které jsou starší než nastavený počet dní.
                                </p>
                            </div>
                        </div>
                    </div>
                </div>

                </div>
       
    `;
    }
    
    /**
     * Vykreslení záložky Advanced
     */
    function renderAdvancedTab() {
        return `
            <div class="card">
                <div class="card-header">
                    WordPress Bloat Removal
                </div>
                <div class="card-body">
                    <form id="bloat-form">
                        <div class="grid md:grid-cols-2 gap-4">
                            <div class="form-group">
                                <label class="form-label">
                                    <input type="checkbox" class="form-checkbox" name="bloat_remove_google_fonts" ${state.config.bloat_remove_google_fonts ? 'checked' : ''}>
                                    Remove Google Fonts
                                </label>
                                <p class="text-sm text-slate-500">Remove Google Fonts to improve privacy and performance.</p>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">
                                    <input type="checkbox" class="form-checkbox" name="bloat_disable_emojis" ${state.config.bloat_disable_emojis ? 'checked' : ''}>
                                    Disable Emojis
                                </label>
                                <p class="text-sm text-slate-500">Remove WordPress emoji scripts and styles.</p>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">
                                    <input type="checkbox" class="form-checkbox" name="bloat_disable_xml_rpc" ${state.config.bloat_disable_xml_rpc ? 'checked' : ''}>
                                    Disable XML-RPC
                                </label>
                                <p class="text-sm text-slate-500">Disable the XML-RPC functionality, which can be a security risk if not used.</p>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">
                                    <input type="checkbox" class="form-checkbox" name="bloat_disable_jquery_migrate" ${state.config.bloat_disable_jquery_migrate ? 'checked' : ''}>
                                    Disable jQuery Migrate
                                </label>
                                <p class="text-sm text-slate-500">Remove jQuery Migrate script on the frontend.</p>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">
                                    <input type="checkbox" class="form-checkbox" name="bloat_disable_rss_feed" ${state.config.bloat_disable_rss_feed ? 'checked' : ''}>
                                    Disable RSS Feeds
                                </label>
                                <p class="text-sm text-slate-500">Disable WordPress RSS feeds.</p>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">
                                    <input type="checkbox" class="form-checkbox" name="bloat_disable_restapi" ${state.config.bloat_disable_restapi ? 'checked' : ''}>
                                    Restrict REST API
                                </label>
                                <p class="text-sm text-slate-500">Restrict REST API to logged-in users only.</p>
                            </div>
                        </div>
                        
                        <h3 class="text-lg font-semibold mt-6 mb-3">WooCommerce Optimizations</h3>
                        
                        <div class="grid md:grid-cols-2 gap-4">
                            <div class="form-group">
                                <label class="form-label">
                                    <input type="checkbox" class="form-checkbox" name="bloat_disable_woo_cart_fragments" ${state.config.bloat_disable_woo_cart_fragments ? 'checked' : ''}>
                                    Disable Cart Fragments
                                </label>
                                <p class="text-sm text-slate-500">Disable WooCommerce cart fragments on non-cart/checkout pages.</p>
                            </div>
                            
                            <div class="form-group">
                                <label class="form-label">
                                    <input type="checkbox" class="form-checkbox" name="bloat_disable_woo_assets" ${state.config.bloat_disable_woo_assets ? 'checked' : ''}>
                                    Optimize WooCommerce Scripts
                                </label>
                                <p class="text-sm text-slate-500">Load WooCommerce scripts and styles only on WooCommerce pages.</p>
                            </div>
                        </div>
                        
                        <button type="submit" class="button button-primary mt-4">Save Advanced Settings</button>
                    </form>
                </div>
            </div>
            
            <div class="card mt-4">
                <div class="card-header">
                    Import/Export Settings
                </div>
                <div class="card-body">
                    <p class="mb-4">Export your current settings or import settings from another installation.</p>
                    
                    <div class="flex flex-col md:flex-row gap-4">
                        <button id="export-settings" class="button button-secondary">Export Settings</button>
                        <div class="relative">
                            <button id="import-settings-btn" class="button button-secondary">Import Settings</button>
                            <input type="file" id="import-settings-file" class="hidden" accept=".json">
                        </div>
                    </div>
                </div>
            </div>
        `;
    }


    function loadActionSchedulerStatus() {
        const asContentElement = document.getElementById('zizi-cache-actionscheduler-status');
        if (!asContentElement) return;
        
        // Show loading indicator
        asContentElement.innerHTML = '<div class="text-center p-4"><div class="spinner" style="display: inline-block;"></div><p>Loading ActionScheduler information...</p></div>';
        
        // Fetch status via AJAX
        fetch(`${restUrl}/actionscheduler-status`, {
            method: 'GET',
            headers: {
                'Accept': 'text/html',
                'X-WP-Nonce': wpApiSettings.nonce
            }
        })
        .then(response => {
            if (!response.ok) {
                throw new Error('Network response was not ok');
            }
            return response.text();
        })
        .then(data => {
            asContentElement.innerHTML = data;
        })
        .catch(error => {
            console.error('Error fetching ActionScheduler status:', error);
            asContentElement.innerHTML = '<div class="bg-red-100 text-red-700 p-4 rounded">An error occurred while loading ActionScheduler status.</div>';
        });
    }    
 // Přidat tento kód po řádce s funkcí loadActionSchedulerStatus

function loadActionSchedulerStatus() {
    const asContentElement = document.getElementById('zizi-cache-actionscheduler-status');
    if (!asContentElement) return;
    
    // Show loading indicator
    asContentElement.innerHTML = '<div class="text-center p-4"><div class="spinner" style="display: inline-block;"></div><p>Načítání informací o ActionScheduler...</p></div>';
    
    // Fetch status via AJAX
    fetch(`${restUrl}/actionscheduler-status`, {
        method: 'GET',
        headers: {
            'Accept': 'text/html',
            'X-WP-Nonce': wpApiSettings.nonce
        }
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.text();
    })
    .then(data => {
        asContentElement.innerHTML = data;
        
        // Přidání event listeneru pro tlačítko čištění
        const runAsCleanupButton = document.getElementById('run-as-cleanup');
        if (runAsCleanupButton) {
            runAsCleanupButton.addEventListener('click', () => {
                runActionSchedulerCleanup();
            });
        }
        
        // Přidání event listeneru pro tlačítko uložení nastavení
        const saveAsSettingsButton = document.getElementById('save-as-settings');
        if (saveAsSettingsButton) {
            saveAsSettingsButton.addEventListener('click', function() {
                saveActionSchedulerSettings();
            });
        }
    })
    .catch(error => {
        console.error('Error fetching ActionScheduler status:', error);
        asContentElement.innerHTML = '<div class="bg-red-100 text-red-700 p-4 rounded">Došlo k chybě při načítání stavu ActionScheduler.</div>';
    });
}

// Přidat funkci pro spuštění čištění ActionScheduler
function runActionSchedulerCleanup() {
    setLoading(true);
    
    fetch(`${restUrl}/clean-actionscheduler`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-WP-Nonce': wpApiSettings.nonce
        }
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.json();
    })
    .then(data => {
        setLoading(false);
        if (data.success) {
            showNotification('Action Scheduler byl úspěšně vyčištěn.', 'success');
            
            // Znovu načtení dat
            loadActionSchedulerStatus();
        } else {
            showNotification('Při čištění Action Scheduler došlo k chybě.', 'error');
        }
    })
    .catch(error => {
        setLoading(false);
        console.error('Error cleaning ActionScheduler:', error);
        showNotification('Při čištění Action Scheduler došlo k chybě.', 'error');
    });
}

// Přidat funkci pro ukládání nastavení ActionScheduler
function saveActionSchedulerSettings() {
    const form = document.getElementById('actionscheduler-cleanup-form');
    if (!form) return;
    
    setLoading(true);
    
    // Získání hodnot z formuláře
    const settings = {
        db_actionscheduler_clean: !!form.querySelector('input[name="db_actionscheduler_clean"]').checked,
        db_actionscheduler_logs_days: parseInt(form.querySelector('input[name="db_actionscheduler_logs_days"]').value) || 1,
        db_actionscheduler_actions_days: parseInt(form.querySelector('input[name="db_actionscheduler_actions_days"]').value) || 7
    };
    
    // Odeslání nastavení na server
    updateConfig(settings, function() {
        showNotification('Nastavení Action Scheduler bylo úspěšně uloženo.', 'success');
    });
}   
    /**
     * Přidání posluchačů událostí
     */
    function addEventListeners() {
        // Záložky
        const tabButtons = document.querySelectorAll('.tab-button');
        tabButtons.forEach(button => {
            button.addEventListener('click', () => {
                const tab = button.dataset.tab;
                changeTab(tab);
            });
        });
   // Add this somewhere in the addEventListeners function
// Load ActionScheduler status when database tab is active
if (state.activeTab === 'database') {
    loadActionSchedulerStatus();
}     
        // Quick Actions na Dashboardu
        const purgeAllButton = document.getElementById('purge-all-cache');
        if (purgeAllButton) {
            purgeAllButton.addEventListener('click', () => {
                purgeEverything();
            });
        }
        
        const purgeAndPreloadButton = document.getElementById('purge-and-preload');
        if (purgeAndPreloadButton) {
            purgeAndPreloadButton.addEventListener('click', () => {
                purgeEverythingAndPreload();
            });
        }
          const preloadCacheButton = document.getElementById('preload-cache');
        if (preloadCacheButton) {
            preloadCacheButton.addEventListener('click', () => {
                preloadCache();
            });
        }
          // Refresh Performance Metrics
        const refreshPerformanceButton = document.getElementById('refresh-performance');
        if (refreshPerformanceButton) {
            refreshPerformanceButton.addEventListener('click', () => {
                fetchPerformanceMetrics();
            });
        }
        
        // Test HTTP Performance
        const testHttpButton = document.getElementById('test-http-performance');
        if (testHttpButton) {
            testHttpButton.addEventListener('click', () => {
                testHttpPerformance();
            });
        }
        
        // Formuláře
        const cachingForm = document.getElementById('caching-form');
        if (cachingForm) {
            cachingForm.addEventListener('submit', (e) => {
                e.preventDefault();
                saveCachingSettings(cachingForm);
            });
        }
        
        const cssOptimizationForm = document.getElementById('css-optimization-form');
        if (cssOptimizationForm) {
            cssOptimizationForm.addEventListener('submit', (e) => {
                e.preventDefault();
                saveCssSettings(cssOptimizationForm);
            });
        }

        const cssRucssCheckbox = document.querySelector('input[name="css_rucss"]');
if (cssRucssCheckbox) {
    cssRucssCheckbox.addEventListener('change', function() {
        const rucssOptions = document.querySelectorAll('.css-rucss-options');
        rucssOptions.forEach(option => {
            option.style.display = this.checked ? 'block' : 'none';
        });
    });
}
        
        const jsOptimizationForm = document.getElementById('js-optimization-form');
        if (jsOptimizationForm) {
            jsOptimizationForm.addEventListener('submit', (e) => {
                e.preventDefault();
                saveJsSettings(jsOptimizationForm);
            });
        }

// Přidání event listeneru pro checkbox js_defer
const jsDeferCheckbox = document.querySelector('input[name="js_defer"]');
if (jsDeferCheckbox) {
    jsDeferCheckbox.addEventListener('change', function() {
        const jsDeferOptions = document.querySelector('.js-defer-options');
        if (jsDeferOptions) {
            jsDeferOptions.style.display = this.checked ? 'block' : 'none';
        }
    });
}

// Přidání event listeneru pro checkbox js_delay
const jsDelayCheckbox = document.querySelector('input[name="js_delay"]');
if (jsDelayCheckbox) {
    jsDelayCheckbox.addEventListener('change', function() {
        const jsDelayOptions = document.querySelector('.js-delay-options');
        if (jsDelayOptions) {
            jsDelayOptions.style.display = this.checked ? 'block' : 'none';
        }
        
        // Aktualizace zobrazení příslušných nastavení podle vybrané metody
        updateJsDelayOptionsVisibility(this.checked);
    });
}



// Přidání event listenerů pro radio buttony js_delay_method
const jsDelayMethodRadios = document.querySelectorAll('input[name="js_delay_method"]');
jsDelayMethodRadios.forEach(radio => {
    radio.addEventListener('change', function() {
        updateJsDelayOptionsVisibility(jsDelayCheckbox ? jsDelayCheckbox.checked : false);
    });
});

// Pomocná funkce pro aktualizaci zobrazení nastavení
function updateJsDelayOptionsVisibility(isChecked) {
    const selectedOptions = document.querySelector('.js-delay-selected-options');
    const allOptions = document.querySelector('.js-delay-all-options');
    
    if (selectedOptions && allOptions) {
        const delayMethodSelected = document.querySelector('input[name="js_delay_method"][value="selected"]:checked');
        if (delayMethodSelected) {
            selectedOptions.style.display = isChecked ? 'block' : 'none';
            allOptions.style.display = 'none';
        } else {
            selectedOptions.style.display = 'none';
            allOptions.style.display = isChecked ? 'block' : 'none';
        }
    }
}
        
        const imgOptimizationForm = document.getElementById('img-optimization-form');
        if (imgOptimizationForm) {
            imgOptimizationForm.addEventListener('submit', (e) => {
                e.preventDefault();
                saveImgSettings(imgOptimizationForm);
            });
        }
        
        const fontOptimizationForm = document.getElementById('font-optimization-form');
        if (fontOptimizationForm) {
            fontOptimizationForm.addEventListener('submit', (e) => {
                e.preventDefault();
                saveFontSettings(fontOptimizationForm);
            });
        }
        
        const cdnForm = document.getElementById('cdn-form');
        if (cdnForm) {
            cdnForm.addEventListener('submit', (e) => {
                e.preventDefault();
                saveCdnSettings(cdnForm);
            });
            
            // Přepínání mezi CDN typy
            const cdnEnabledCheckbox = cdnForm.querySelector('input[name="cdn"]');
            const cdnTypeSelect = cdnForm.querySelector('#cdn_type');
            const customCdnSettings = cdnForm.querySelector('#custom-cdn-settings');
            const ziziCdnSettings = cdnForm.querySelector('#zizi-cdn-settings');
            const cdnFileTypesSelect = cdnForm.querySelector('#cdn_file_types');
            
            if (cdnEnabledCheckbox && cdnTypeSelect && customCdnSettings && ziziCdnSettings && cdnFileTypesSelect) {
                cdnEnabledCheckbox.addEventListener('change', () => {
                    const isEnabled = cdnEnabledCheckbox.checked;
                    cdnTypeSelect.disabled = !isEnabled;
                    cdnFileTypesSelect.disabled = !isEnabled;
                    
                    if (!isEnabled) {
                        customCdnSettings.style.display = 'none';
                        ziziCdnSettings.style.display = 'none';
                    } else {
                        updateCdnTypeVisibility(cdnTypeSelect.value);
                    }
                });
                
                cdnTypeSelect.addEventListener('change', () => {
                    updateCdnTypeVisibility(cdnTypeSelect.value);
                });
                
                function updateCdnTypeVisibility(type) {
                    if (type === 'custom') {
                        customCdnSettings.style.display = 'block';
                        ziziCdnSettings.style.display = 'none';
                    } else if (type === 'zizi_cdn') {
                        customCdnSettings.style.display = 'none';
                        ziziCdnSettings.style.display = 'block';
                    }
                }
            }
        }
        
        const databaseForm = document.getElementById('database-form');
        if (databaseForm) {
            databaseForm.addEventListener('submit', (e) => {
                e.preventDefault();
                saveDatabaseSettings(databaseForm);
            });
            
            const runDbCleanupButton = document.getElementById('run-db-cleanup');
            if (runDbCleanupButton) {
                runDbCleanupButton.addEventListener('click', () => {
                    runDatabaseCleanup();
                });
            }
        }
        
        const bloatForm = document.getElementById('bloat-form');
        if (bloatForm) {
            bloatForm.addEventListener('submit', (e) => {
                e.preventDefault();
                saveBloatSettings(bloatForm);
            });
        }
        
        // Import/Export
        const exportSettingsButton = document.getElementById('export-settings');
        if (exportSettingsButton) {
            exportSettingsButton.addEventListener('click', () => {
                exportSettings();
            });
        }
        
        const importSettingsButton = document.getElementById('import-settings-btn');
        const importSettingsFile = document.getElementById('import-settings-file');
        if (importSettingsButton && importSettingsFile) {
            importSettingsButton.addEventListener('click', () => {
                importSettingsFile.click();
            });
            
            importSettingsFile.addEventListener('change', (e) => {
                importSettings(e.target.files[0]);
            });
        }

// Event listener pro ActionScheduler cleanup
const asContentElement = document.getElementById('zizi-cache-actionscheduler-status');
if (asContentElement) {
    // Zobrazíme indikátor načítání
    asContentElement.innerHTML = '<div class="text-center p-4"><div class="spinner" style="display: inline-block;"></div><p>Načítání informací o ActionScheduler...</p></div>';
    
    // Načtení obsahu přes AJAX
    fetch(`${restUrl}/actionscheduler-status`, {
        method: 'GET',
        headers: {
            'Accept': 'text/html',
            'X-WP-Nonce': wpApiSettings.nonce
        }
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.text();
    })
    .then(data => {
        asContentElement.innerHTML = data;
        
        // Přidání event listeneru pro tlačítko čištění
        const runAsCleanupButton = document.getElementById('run-as-cleanup');
        if (runAsCleanupButton) {
            runAsCleanupButton.addEventListener('click', () => {
                runActionSchedulerCleanup();
            });
        }
    })
    .catch(error => {
        console.error('Error fetching ActionScheduler status:', error);
        asContentElement.innerHTML = '<div class="bg-red-100 text-red-700 p-4 rounded">Došlo k chybě při načítání stavu ActionScheduler.</div>';
    });
}
    }
    
/**
 * Uložení CSS nastavení
 */
function saveCssSettings(form) {
    setLoading(true);
    
    const settings = {
        css_minify: !!form.querySelector('input[name="css_minify"]').checked,
        css_rucss: !!form.querySelector('input[name="css_rucss"]').checked
    };
    
    // Získání metody pro načítání nepoužitého CSS
    if (settings.css_rucss) {
        const rucssMethod = form.querySelector('input[name="css_rucss_method"]:checked');
        settings.css_rucss_method = rucssMethod ? rucssMethod.value : 'async';
        
        // Zpracování textových polí pro RUCSS
        const excludeStylesheets = form.querySelector('#css_rucss_exclude_stylesheets').value.trim();
        settings.css_rucss_exclude_stylesheets = excludeStylesheets ? 
            excludeStylesheets.split('\n').map(url => url.trim()).filter(url => url) : [];
        
        const includeSelectors = form.querySelector('#css_rucss_include_selectors').value.trim();
        settings.css_rucss_include_selectors = includeSelectors ? 
            includeSelectors.split('\n').map(selector => selector.trim()).filter(selector => selector) : [];
    }
    
    updateConfig(settings);
}

/**
 * Uložení JavaScript nastavení
 */
function saveJsSettings(form) {
    setLoading(true);
    
    // Základní nastavení
    // Základní nastavení
    const settings = {
        js_minify: !!form.querySelector('input[name="js_minify"]').checked,
        js_defer: !!form.querySelector('input[name="js_defer"]').checked,
        js_delay: !!form.querySelector('input[name="js_delay"]').checked,
        self_host_third_party_css_js: !!form.querySelector('input[name="host_third_party_locally"]').checked,
        lazy_render_enabled: !!form.querySelector('input[name="lazy_render_enabled"]').checked
    };
    
    // Zpracování textarea pro js_defer_excludes
    if (settings.js_defer) {
        const jsDeferExcludesTextarea = form.querySelector('#js_defer_excludes');
        if (jsDeferExcludesTextarea) {
            const deferExcludeScripts = jsDeferExcludesTextarea.value.trim();
            settings.js_defer_excludes = deferExcludeScripts ? 
                deferExcludeScripts.split('\n').map(s => s.trim()).filter(s => s) : [];
        }
    }
    
    // Zpracování metody delay
    if (settings.js_delay) {
        const jsDelayMethodRadio = form.querySelector('input[name="js_delay_method"]:checked');
        settings.js_delay_method = jsDelayMethodRadio ? jsDelayMethodRadio.value : 'selected';
        
        // Důležité: získat hodnotu textarey podle ID a správně ji zpracovat
        const jsDelaySelectedTextarea = form.querySelector('#js_delay_selected');
        if (jsDelaySelectedTextarea) {
            const selectedScripts = jsDelaySelectedTextarea.value.trim();
            settings.js_delay_selected = selectedScripts ? 
                selectedScripts.split('\n').map(s => s.trim()).filter(s => s) : [];
            
            // Debug - přidejte dočasně pro kontrolu
            console.log('Saving js_delay_selected:', settings.js_delay_selected);
        }
        
        // Důležité: získat hodnotu textarey podle ID a správně ji zpracovat
        const jsDelayAllExcludesTextarea = form.querySelector('#js_delay_all_excludes');
        if (jsDelayAllExcludesTextarea) {
            const excludeScripts = jsDelayAllExcludesTextarea.value.trim();
            settings.js_delay_all_excludes = excludeScripts ? 
                excludeScripts.split('\n').map(s => s.trim()).filter(s => s) : [];
            
            // Debug - přidejte dočasně pro kontrolu
            console.log('Saving js_delay_all_excludes:', settings.js_delay_all_excludes);
        }
        
        const jsDelayExcludesTextarea = form.querySelector('#js_delay_excludes');
        if (jsDelayExcludesTextarea) {
            const excludeDelayScripts = jsDelayExcludesTextarea.value.trim();
            settings.js_delay_excludes = excludeDelayScripts ? 
                excludeDelayScripts.split('\n').map(s => s.trim()).filter(s => s) : [];
        }
    }
    
    // Zpracování seznamu pro lazy render selectors
    const lazyRenderSelectors = form.querySelector('#lazy_render_selectors');
    if (lazyRenderSelectors) {
        const selectors = lazyRenderSelectors.value.trim();
        settings.js_lazy_render_selectors = selectors ? 
            selectors.split('\n').map(s => s.trim()).filter(s => s) : [];
    }
    
    updateConfig(settings);
}
    /**
     * Přepnutí záložky
     */
    function changeTab(tab) {
        state.activeTab = tab;
        
        // Aktualizace UI
        const tabButtons = document.querySelectorAll('.tab-button');
        tabButtons.forEach(button => {
            if (button.dataset.tab === tab) {
                button.classList.add('active');
            } else {
                button.classList.remove('active');
            }
            // At the end of the changeTab function, add:
// Load ActionScheduler status if database tab is active
if (tab === 'database') {
    // Počkáme na dokončení vykreslení obsahu a pak načteme stav Action Scheduler
    setTimeout(() => {
        loadActionSchedulerStatus();
    }, 100);
}
        });
          // Aktualizace obsahu
        const tabContent = document.getElementById('tab-content');
        tabContent.innerHTML = renderTabContent();
        
        // Znovu přidáme posluchače událostí
        addEventListeners();
        
        // Aktualizace performance metrik při přechodu na dashboard
        if (tab === 'dashboard') {
            setTimeout(() => {
                updatePerformanceDisplay();
            }, 100);
        }
    }
    
    /**
     * Získání počtu cacheovaných stránek
     */
    function fetchCachedPagesCount() {
        fetch(`${restUrl}/cached-pages-count`, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-WP-Nonce': wpApiSettings.nonce
            }
        })
        .then(response => response.json())
        .then(data => {
            state.cachedPagesCount = data.count;
            const cachedPagesCountElement = document.getElementById('cached-pages-count');
            if (cachedPagesCountElement) {
                cachedPagesCountElement.textContent = data.count;
            }
        })
        .catch(error => {
            console.error('Error fetching cached pages count:', error);
        });    }
    
    /**
     * Načtení performance metrik z API
     */
    function fetchPerformanceMetrics() {
        state.performanceLoading = true;
        
        fetch(`${restUrl}/performance-metrics`, {
            method: 'GET',
            headers: {
                'Content-Type': 'application/json',
                'X-WP-Nonce': wpApiSettings.nonce
            }
        })
        .then(response => response.json())
        .then(data => {
            state.performanceMetrics = data.current_metrics;
            state.performanceLoading = false;
            updatePerformanceDisplay();
        })
        .catch(error => {
            console.error('Error fetching performance metrics:', error);
            state.performanceLoading = false;
            state.performanceMetrics = null;
            updatePerformanceDisplay();
        });
    }    /**
     * Aktualizace zobrazení performance metrik
     */
    function updatePerformanceDisplay() {
        if (!state.performanceMetrics) {
            return;
        }
        
        const db = state.performanceMetrics.database_performance || {};
        const http = state.performanceMetrics.http_performance || {};
        const cache = state.performanceMetrics.cache_efficiency || {};
        
        // Update database performance - nový user-friendly formát
        updatePerformanceCard('database', db);
        updatePerformanceCard('http', http);
        updatePerformanceCard('cache', cache);
    }
    
    /**
     * Aktualizace jednotlivé performance karty
     */
    function updatePerformanceCard(type, data) {
        const card = document.querySelector(`[data-performance-card="${type}"]`);
        if (!card) return;
        
        // Najdi nebo vytvoř elementy
        let statusElement = card.querySelector('.performance-status');
        let messageElement = card.querySelector('.performance-message');
        let descriptionElement = card.querySelector('.performance-description');
        let recommendationElement = card.querySelector('.performance-recommendation');
        let iconElement = card.querySelector('.performance-icon');
        
        if (!statusElement) {
            // Vytvoř novou strukturu pro user-friendly zobrazení
            card.innerHTML = `
                <div class="performance-card-header" style="display: flex; align-items: center; margin-bottom: 12px;">
                    <span class="performance-icon" style="font-size: 24px; margin-right: 8px;">${data.icon || '📊'}</span>
                    <div>
                        <h4 class="performance-message" style="margin: 0; color: ${data.color || '#000'}; font-weight: 600;">${data.message || 'Načítání...'}</h4>
                        <span class="performance-status" style="font-size: 12px; text-transform: uppercase; opacity: 0.7;">${getPerformanceTypeTitle(type)}</span>
                    </div>
                </div>
                <p class="performance-description" style="margin: 0 0 8px 0; color: #666; font-size: 14px;">${data.description || ''}</p>
                <div class="performance-recommendation" style="font-size: 13px; color: #f59e0b; font-weight: 500;">${data.recommendation || ''}</div>
                ${data.technical_details ? `
                <details style="margin-top: 12px; font-size: 12px; opacity: 0.7;">
                    <summary style="cursor: pointer;">Technické detaily</summary>
                    <div style="margin-top: 4px; padding: 8px; background: #f9fafb; border-radius: 4px;">
                        ${formatTechnicalDetails(data.technical_details)}
                    </div>
                </details>
                ` : ''}
            `;
            return;
        }
        
        // Aktualizuj existující elementy
        if (iconElement) iconElement.textContent = data.icon || '📊';
        if (messageElement) {
            messageElement.textContent = data.message || 'Načítání...';
            messageElement.style.color = data.color || '#000';
        }
        if (descriptionElement) descriptionElement.textContent = data.description || '';
        if (recommendationElement) recommendationElement.textContent = data.recommendation || '';
    }
    
    /**
     * Získá název typu performance metriky
     */
    function getPerformanceTypeTitle(type) {
        const titles = {
            'database': 'Databázový výkon',
            'http': 'Síťové připojení', 
            'cache': 'Cache systém'
        };
        return titles[type] || type;
    }
    
    /**
     * Formátuje technické detaily pro zobrazení
     */
    function formatTechnicalDetails(details) {
        if (!details || typeof details !== 'object') return '';
        
        return Object.entries(details).map(([key, value]) => {
            const labels = {
                'queries': 'Databázové dotazy',
                'memory_mb': 'Paměť (MB)',
                'peak_memory_mb': 'Max paměť (MB)',
                'requests': 'HTTP požadavky',
                'avg_response_ms': 'Průměrná odezva (ms)',
                'error_rate': 'Chybovost (%)',
                'opcache_enabled': 'OPcache',
                'object_cache_enabled': 'Object cache',
                'zizi_cache_active': 'ZiziCache'
            };
            
            const label = labels[key] || key;
            let displayValue = value;
            
            if (typeof value === 'boolean') {
                displayValue = value ? '✅ Ano' : '❌ Ne';
            }
            
            return `<div><strong>${label}:</strong> ${displayValue}</div>`;
        }).join('');
    }
      /**
     * Test HTTP Performance - vyvolá HTTP požadavky pro testování metrik
     */
    function testHttpPerformance() {
        showNotification('Testuji síťové připojení... Může to trvat několik sekund.', 'info');
        
        // Vytvoříme skrytý iframe, který načte stránku, která vyvolá HTTP požadavky
        const iframe = document.createElement('iframe');
        iframe.style.display = 'none';
        iframe.src = `${window.location.origin}/wp-admin/admin-ajax.php?action=zizi_test_http_performance&nonce=${wpApiSettings.nonce}`;
        
        iframe.onload = function() {
            // Po 3 sekundách odstraníme iframe a refreshneme metriky
            setTimeout(() => {
                document.body.removeChild(iframe);
                fetchPerformanceMetrics();
                showNotification('Test síťového připojení dokončen. Metriky obnoveny.', 'success');
            }, 3000);
        };
        
        iframe.onerror = function() {
            document.body.removeChild(iframe);
            showNotification('Test síťového připojení selhal.', 'error');
        };
        
        document.body.appendChild(iframe);
    }
    
    /**
     * Vymazání kompletní cache
     */
    function purgeEverything() {
        if (!confirm('Are you sure you want to purge all cache?')) {
            return;
        }
        
        setLoading(true);
        
        fetch(`${restUrl}/purge-pages`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-WP-Nonce': wpApiSettings.nonce
            }
        })
        .then(response => response.json())
        .then(data => {
            setLoading(false);
            if (data.success) {
                showNotification('All cache has been purged successfully.', 'success');
                fetchCachedPagesCount();
            } else {
                showNotification('Failed to purge cache.', 'error');
            }
        })
        .catch(error => {
            setLoading(false);
            console.error('Error purging cache:', error);
            showNotification('An error occurred while purging cache.', 'error');
        });
    }
    
    /**
     * Vymazání cache a spuštění preload
     */
    function purgeEverythingAndPreload() {
        if (!confirm('Are you sure you want to purge all cache and preload?')) {
            return;
        }
        
        setLoading(true);
        
        fetch(`${restUrl}/purge-everything-and-preload`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-WP-Nonce': wpApiSettings.nonce
            }
        })
        .then(response => response.json())
        .then(data => {
            setLoading(false);
            if (data.success) {
                showNotification('All cache has been purged and preload started.', 'success');
                fetchCachedPagesCount();
            } else {
                showNotification('Failed to purge cache and preload.', 'error');
            }
        })
        .catch(error => {
            setLoading(false);
            console.error('Error purging cache and preloading:', error);
            showNotification('An error occurred while purging cache and preloading.', 'error');
        });
    }
    
    /**
     * Spuštění preload cache
     */
    function preloadCache() {
        setLoading(true);
        
        fetch(`${restUrl}/preload-cache`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-WP-Nonce': wpApiSettings.nonce
            }
        })
        .then(response => response.json())
        .then(data => {
            setLoading(false);
            if (data.success) {
                showNotification('Cache preload has been started.', 'success');
            } else {
                showNotification('Failed to start cache preload.', 'error');
            }
        })
        .catch(error => {
            setLoading(false);
            console.error('Error preloading cache:', error);
            showNotification('An error occurred while starting cache preload.', 'error');
        });
    }
    
    /**
     * Spuštění čištění databáze
     */
    function runDatabaseCleanup() {
        setLoading(true);
        
        // Získání nastavení z formuláře
        const dbForm = document.getElementById('database-form');
        if (!dbForm) {
            setLoading(false);
            return;
        }
        
        const settings = getFormData(dbForm);
        
        // Přidání nastavení pro ActionScheduler
        settings.db_actionscheduler_clean = !!dbForm.querySelector('input[name="db_actionscheduler_clean"]').checked;
        
        const logsdays = dbForm.querySelector('#db_actionscheduler_logs_days');
        settings.db_actionscheduler_logs_days = logsdays ? parseInt(logsdays.value) || 1 : 1;
        
        const actionsdays = dbForm.querySelector('#db_actionscheduler_actions_days');
        settings.db_actionscheduler_actions_days = actionsdays ? parseInt(actionsdays.value) || 7 : 7;
        
        // Aktualizace konfigurace a provedení čištění
        updateConfig(settings, () => {
            // Po kompletní aktualizaci nastavení provedeme čištění na serveru
            fetch(`${restUrl}/clean-database`, {
                method: 'POST',
                headers: {
                    'Content-Type': 'application/json',
                    'X-WP-Nonce': wpApiSettings.nonce
                }
            })
            .then(response => response.json())
            .then(data => {
                // Vždy spustíme čištění ActionScheduler, pokud je povoleno
                if (settings.db_actionscheduler_clean) {
                    return fetch(`${restUrl}/clean-actionscheduler`, {
                        method: 'POST',
                        headers: {
                            'Content-Type': 'application/json',
                            'X-WP-Nonce': wpApiSettings.nonce
                        }
                    })
                    .then(response => response.json());
                }
                return data;
            })
            .then(() => {
                setLoading(false);
                showNotification('Databázové čištění bylo úspěšně dokončeno.', 'success');
                
                // Aktualizace stavu ActionScheduler, pokud je zobrazen
                if (document.getElementById('actionscheduler-status-content')) {
                    loadActionSchedulerStatus();
                }
            })
            .catch(error => {
                setLoading(false);
                console.error('Chyba při čištění databáze:', error);
                showNotification('Při čištění databáze došlo k chybě.', 'error');
            });
        });
    }

/**
 * Spuštění čištění ActionScheduler
 */
function runActionSchedulerCleanup() {
    setLoading(true);
    
    fetch(`${restUrl}/clean-actionscheduler`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-WP-Nonce': wpApiSettings.nonce
        }
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.json();
    })
    .then(data => {
        setLoading(false);
        if (data.success) {
            showNotification('Action Scheduler byl úspěšně vyčištěn.', 'success');
            
            // Znovu načtení dat
            const asContentElement = document.getElementById('zizi-cache-actionscheduler-status');
            if (asContentElement) {
                asContentElement.innerHTML = '<div class="text-center p-4"><div class="spinner" style="display: inline-block;"></div><p>Aktualizace informací o ActionScheduler...</p></div>';
                
                fetch(`${restUrl}/actionscheduler-status`, {
                    method: 'GET',
                    headers: {
                        'Accept': 'text/html',
                        'X-WP-Nonce': wpApiSettings.nonce
                    }
                })
                .then(response => response.text())
                .then(content => {
                    asContentElement.innerHTML = content;
                    
                    // Přidání event listeneru pro tlačítko po aktualizaci obsahu
                    const runAsCleanupButton = document.getElementById('run-as-cleanup');
                    if (runAsCleanupButton) {
                        runAsCleanupButton.addEventListener('click', () => {
                            runActionSchedulerCleanup();
                        });
                    }
                })
                .catch(error => {
                    console.error('Error refreshing ActionScheduler status:', error);
                    asContentElement.innerHTML = '<div class="bg-red-100 text-red-700 p-4 rounded">Došlo k chybě při aktualizaci stavu ActionScheduler.</div>';
                });
            }
        } else {
            showNotification('Při čištění Action Scheduler došlo k chybě.', 'error');
        }
    })
    .catch(error => {
        setLoading(false);
        console.error('Error cleaning ActionScheduler:', error);
        showNotification('Při čištění Action Scheduler došlo k chybě.', 'error');
    });
}

 /**
 * Spuštění čištění ActionScheduler
 */
function runActionSchedulerCleanup() {
    setLoading(true);
    
    fetch(`${restUrl}/clean-actionscheduler`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-WP-Nonce': wpApiSettings.nonce
        }
    })
    .then(response => response.json())
    .then(data => {
        setLoading(false);
        if (data.success) {
            showNotification('Action Scheduler byl úspěšně vyčištěn.', 'success');
            
            // Znovu načtení dat
            const asContentElement = document.getElementById('zizi-cache-actionscheduler-status');
            if (asContentElement) {
                fetch(`${restUrl}/actionscheduler-status`, {
                    method: 'GET',
                    headers: {
                        'Content-Type': 'application/json',
                        'X-WP-Nonce': wpApiSettings.nonce
                    }
                })
                .then(response => response.text())
                .then(content => {
                    asContentElement.innerHTML = content;
                })
                .catch(error => {
                    console.error('Error refreshing ActionScheduler status:', error);
                });
            }
        } else {
            showNotification('Při čištění Action Scheduler došlo k chybě.', 'error');
        }
    })
    .catch(error => {
        setLoading(false);
        console.error('Error cleaning ActionScheduler:', error);
        showNotification('Při čištění Action Scheduler došlo k chybě.', 'error');
    });
}   
    
/**
 * Uložení caching nastavení
 */
function saveCachingSettings(form) {
    setLoading(true);
    
    const settings = getFormData(form);
    
    // Zpracování checkboxů
    settings.cache_logged_in = !!form.querySelector('input[name="cache_logged_in"]').checked;
    settings.cache_mobile = !!form.querySelector('input[name="cache_mobile"]').checked;
    settings.cache_preload = !!form.querySelector('input[name="cache_preload"]').checked;
    
    // Zpracování textových polí pro Exclude pages from cache
    const excludePages = form.querySelector('#cache_exclude_pages').value.trim();
    settings.cache_exclude_pages = excludePages ? excludePages.split('\n').map(url => url.trim()).filter(url => url) : [];
    
    // Zpracování textových polí pro Ignore query parameters
    const ignoreQueries = form.querySelector('#cache_ignore_queries').value.trim();
    settings.cache_ignore_queries = ignoreQueries ? ignoreQueries.split('\n').map(param => param.trim()).filter(param => param) : [];
    
    // Zpracování textových polí pro Include query parameters
    const includeQueries = form.querySelector('#cache_include_queries').value.trim();
    settings.cache_include_queries = includeQueries ? includeQueries.split('\n').map(param => param.trim()).filter(param => param) : [];
    
    // Zpracování textových polí pro Bypass cookies
    const bypassCookies = form.querySelector('#cache_bypass_cookies').value.trim();
    settings.cache_bypass_cookies = bypassCookies ? bypassCookies.split('\n').map(cookie => cookie.trim()).filter(cookie => cookie) : [];
    
    updateConfig(settings);
}
    
    /**
     * Uložení JavaScript nastavení
     */
    function saveJsSettings(form) {
        setLoading(true);
        
        const settings = {
            js_minify: !!form.querySelector('input[name="js_minify"]').checked,
            js_defer: !!form.querySelector('input[name="js_defer"]').checked,
            js_delay: !!form.querySelector('input[name="js_delay"]').checked,
            self_host_third_party_css_js: !!form.querySelector('input[name="host_third_party_locally"]').checked,
            lazy_render_enabled: !!form.querySelector('input[name="lazy_render_enabled"]').checked
        };
        
        // Zpracování textarea pro js_defer_excludes
        const jsDeferExcludesTextarea = form.querySelector('#js_defer_excludes');
        if (jsDeferExcludesTextarea) {
            const deferExcludeScripts = jsDeferExcludesTextarea.value.trim();
            settings.js_defer_excludes = deferExcludeScripts ? 
                deferExcludeScripts.split('\n').map(s => s.trim()).filter(s => s) : [];
        }
        
        // Process delay method and textarea values
        if (settings.js_delay) {
            // Get delay method
            const jsDelayMethodRadio = form.querySelector('input[name="js_delay_method"]:checked');
            settings.js_delay_method = jsDelayMethodRadio ? jsDelayMethodRadio.value : 'selected';
            
            // Get selected scripts
            const jsDelaySelectedTextarea = form.querySelector('#js_delay_selected');
            if (jsDelaySelectedTextarea) {
                const selectedScripts = jsDelaySelectedTextarea.value.trim();
                settings.js_delay_selected = selectedScripts ? 
                    selectedScripts.split('\n').map(s => s.trim()).filter(s => s) : [];
            }
            
            // Get excluded scripts for "all" method
            const jsDelayAllExcludesTextarea = form.querySelector('#js_delay_all_excludes');
            if (jsDelayAllExcludesTextarea) {
                const excludeScripts = jsDelayAllExcludesTextarea.value.trim();
                settings.js_delay_all_excludes = excludeScripts ? 
                    excludeScripts.split('\n').map(s => s.trim()).filter(s => s) : [];
            }
        }
        
        // Get lazy render selectors
        const lazyRenderSelectors = form.querySelector('#lazy_render_selectors');
        if (lazyRenderSelectors) {
            const selectors = lazyRenderSelectors.value.trim();
            settings.js_lazy_render_selectors = selectors ? 
                selectors.split('\n').map(s => s.trim()).filter(s => s) : [];
        }
        
        updateConfig(settings);
    }
    
    /**
     * Uložení Image nastavení
     */
    function saveImgSettings(form) {
        setLoading(true);
        
        const settings = {
            img_lazyload: !!form.querySelector('input[name="img_lazyload"]').checked,
            img_responsive: !!form.querySelector('input[name="img_responsive"]').checked,
            img_width_height: !!form.querySelector('input[name="img_width_height"]').checked
        };
        
        updateConfig(settings);
    }
      /**
     * Uložení Font nastavení
     */
    function saveFontSettings(form) {
        setLoading(true);
        
        // Get font ATF height and validate (default to 1200 if invalid)
        const atfHeight = parseInt(form.querySelector('input[name="fonts_atf_height"]').value, 10);
        const validAtfHeight = !isNaN(atfHeight) && atfHeight >= 100 && atfHeight <= 10000 ? atfHeight : 1200;
        
        // Get cache TTL and validate (default to 3600 if invalid)
        const cacheTtl = parseInt(form.querySelector('input[name="fonts_cache_ttl"]').value, 10);
        const validCacheTtl = !isNaN(cacheTtl) && cacheTtl >= 60 && cacheTtl <= 2592000 ? cacheTtl : 3600;
        
        const settings = {
            fonts_optimize_google_fonts: !!form.querySelector('input[name="fonts_optimize_google_fonts"]').checked,
            fonts_display_swap: !!form.querySelector('input[name="fonts_display_swap"]').checked,
            fonts_allow_cdn_domains: !!form.querySelector('input[name="fonts_allow_cdn_domains"]').checked,
            fonts_debug: !!form.querySelector('input[name="fonts_debug"]').checked,
            fonts_atf_height: validAtfHeight,
            fonts_cache_ttl: validCacheTtl
        };
        
        updateConfig(settings);
    }
    
    /**
     * Uložení CDN nastavení
     */
    function saveCdnSettings(form) {
        setLoading(true);
        
        const cdnEnabled = !!form.querySelector('input[name="cdn"]').checked;
        const cdnType = form.querySelector('#cdn_type').value;
        const cdnUrl = form.querySelector('#cdn_url').value.trim();
        const ziziCdnApiKey = form.querySelector('#zizi_cdn_api_key').value.trim();
        const cdnFileTypes = form.querySelector('#cdn_file_types').value;
        
        const settings = {
            cdn: cdnEnabled,
            cdn_type: cdnType,
            cdn_url: cdnUrl,
            zizi_cdn_api_key: ziziCdnApiKey,
            cdn_file_types: cdnFileTypes
        };
        
        updateConfig(settings);
    }
    
    /**
     * Uložení Database nastavení
     */
    function saveDatabaseSettings(form) {
        setLoading(true);
        
        const settings = {
            db_post_revisions: !!form.querySelector('input[name="db_post_revisions"]').checked,
            db_post_auto_drafts: !!form.querySelector('input[name="db_post_auto_drafts"]').checked,
            db_post_trashed: !!form.querySelector('input[name="db_post_trashed"]').checked,
            db_comments_spam: !!form.querySelector('input[name="db_comments_spam"]').checked,
            db_comments_trashed: !!form.querySelector('input[name="db_comments_trashed"]').checked,
            db_transients_expired: !!form.querySelector('input[name="db_transients_expired"]').checked,
            db_transients_all: !!form.querySelector('input[name="db_transients_all"]').checked,
            db_optimize_tables: !!form.querySelector('input[name="db_optimize_tables"]').checked,
            db_schedule_clean: form.querySelector('#db_schedule_clean').value
            
        };
        settings.db_actionscheduler_clean = !!form.querySelector('input[name="db_actionscheduler_clean"]').checked;

        // Zpracování číselných hodnot
        const logsdays = form.querySelector('#db_actionscheduler_logs_days');
        settings.db_actionscheduler_logs_days = logsdays ? parseInt(logsdays.value) || 1 : 1;
        
        const actionsdays = form.querySelector('#db_actionscheduler_actions_days');
        settings.db_actionscheduler_actions_days = actionsdays ? parseInt(actionsdays.value) || 7 : 7;     
        updateConfig(settings);
    }
    
    /**
     * Uložení Bloat nastavení
     */
    function saveBloatSettings(form) {
        setLoading(true);
        
        const settings = {
            bloat_remove_google_fonts: !!form.querySelector('input[name="bloat_remove_google_fonts"]').checked,
            bloat_disable_emojis: !!form.querySelector('input[name="bloat_disable_emojis"]').checked,
            bloat_disable_xml_rpc: !!form.querySelector('input[name="bloat_disable_xml_rpc"]').checked,
            bloat_disable_jquery_migrate: !!form.querySelector('input[name="bloat_disable_jquery_migrate"]').checked,
            bloat_disable_rss_feed: !!form.querySelector('input[name="bloat_disable_rss_feed"]').checked,
            bloat_remove_restapi: !!form.querySelector('input[name="bloat_disable_restapi"]').checked,
            bloat_disable_woo_cart_fragments: !!form.querySelector('input[name="bloat_disable_woo_cart_fragments"]').checked,
            bloat_disable_woo_assets: !!form.querySelector('input[name="bloat_disable_woo_assets"]').checked
        };
        
        updateConfig(settings);
    }
    
    /**
     * Export nastavení
     */
    function exportSettings() {
        const settings = JSON.stringify(state.config, null, 2);
        const blob = new Blob([settings], { type: 'application/json' });
        const url = URL.createObjectURL(blob);
        
        const a = document.createElement('a');
        a.href = url;
        a.download = 'zizi-cache-settings.json';
        document.body.appendChild(a);
        a.click();
        
        // Cleanup
        setTimeout(() => {
            document.body.removeChild(a);
            URL.revokeObjectURL(url);
        }, 100);
    }
    
    /**
     * Import nastavení
     */
    function importSettings(file) {
        if (!file) {
            return;
        }
        
        const reader = new FileReader();
        
        reader.onload = function(e) {
            try {
                const settings = JSON.parse(e.target.result);
                
                if (!settings || typeof settings !== 'object') {
                    showNotification('Invalid settings file.', 'error');
                    return;
                }
                
                // Confirm import
                if (confirm('Are you sure you want to import these settings? This will overwrite your current configuration.')) {
                    setLoading(true);
                    updateConfig(settings);
                }
            } catch (error) {
                console.error('Error parsing settings file:', error);
                showNotification('Failed to parse settings file.', 'error');
            }
        };
        
        reader.readAsText(file);
    }
    
    /**
     * Aktualizace konfigurace
     */
    function updateConfig(settings, callback) {
        fetch(`${restUrl}/config`, {
            method: 'POST',
            headers: {
                'Content-Type': 'application/json',
                'X-WP-Nonce': wpApiSettings.nonce
            },
            body: JSON.stringify(settings)
        })
        .then(response => response.json())
        .then(data => {
            setLoading(false);
            
            if (data.code && data.message) {
                // Error response
                showNotification(data.message, 'error');
            } else {
                // Success response
                state.config = data;
                showNotification('Settings have been saved successfully.', 'success');
                
                if (typeof callback === 'function') {
                    callback();
                }
            }
        })
        .catch(error => {
            setLoading(false);
            console.error('Error updating config:', error);
            showNotification('An error occurred while saving settings.', 'error');
        });
    }
    
    /**
     * Získání dat z formuláře
     */
    function getFormData(form) {
        const formData = new FormData(form);
        const data = {};
        
        for (const [key, value] of formData.entries()) {
            // Převedeme checkboxy na boolean
            if (form.querySelector(`input[name="${key}"][type="checkbox"]`)) {
                data[key] = value === 'on';
            } else {
                data[key] = value;
            }
        }
        
        return data;
    }
    
    /**
     * Zobrazení loading stavu
     */
    function setLoading(isLoading) {
        state.isLoading = isLoading;
        
        // Přidáme nebo odstraníme spinner
        const spinnerContainer = document.createElement('div');
        spinnerContainer.id = 'spinner-overlay';
        spinnerContainer.style.position = 'fixed';
        spinnerContainer.style.top = '0';
        spinnerContainer.style.left = '0';
        spinnerContainer.style.width = '100%';
        spinnerContainer.style.height = '100%';
        spinnerContainer.style.backgroundColor = 'rgba(255, 255, 255, 0.7)';
        spinnerContainer.style.display = 'flex';
        spinnerContainer.style.justifyContent = 'center';
        spinnerContainer.style.alignItems = 'center';
        spinnerContainer.style.zIndex = '9999';
        
        const spinner = document.createElement('div');
        spinner.className = 'spinner';
        spinner.style.width = '50px';
        spinner.style.height = '50px';
        spinner.style.border = '5px solid rgba(79, 70, 229, 0.2)';
        spinner.style.borderTopColor = '#4F46E5';
        
        spinnerContainer.appendChild(spinner);
        
        if (isLoading) {
            // Přidáme spinner
            document.body.appendChild(spinnerContainer);
        } else {
            // Odstraníme spinner
            const existingSpinner = document.getElementById('spinner-overlay');
            if (existingSpinner) {
                document.body.removeChild(existingSpinner);
            }
        }
    }
    
    /**
     * Zobrazení notifikace
     */
    function showNotification(message, type = 'success') {
        state.notification = { message, type };
        
        const notificationArea = document.getElementById('notification-area');
        if (!notificationArea) return;
        
        notificationArea.innerHTML = '';
        notificationArea.classList.remove('hidden');
        
        const notification = document.createElement('div');
        notification.className = `notification notification-${type}`;
        notification.innerHTML = `
            <div class="flex items-center justify-between">
                <div>${message}</div>
                <button type="button" class="notification-close">×</button>
            </div>
        `;
        
        notificationArea.appendChild(notification);
        
        // Přidáme tlačítko pro zavření
        const closeButton = notification.querySelector('.notification-close');
        if (closeButton) {
            closeButton.addEventListener('click', () => {
                notificationArea.classList.add('hidden');
                state.notification = null;
            });
        }
        
        // Automaticky skryjeme po 5 sekundách
        setTimeout(() => {
            if (notificationArea.contains(notification)) {
                notificationArea.classList.add('hidden');
                state.notification = null;
            }
        }, 5000);
    }
/**
 * Načtení stavu ActionScheduler tabulek
 */
function loadActionSchedulerStatus() {
    const loadingElement = document.getElementById('actionscheduler-status-loading');
    const contentElement = document.getElementById('actionscheduler-status-content');
    
    if (!loadingElement || !contentElement) return;
    
    loadingElement.classList.remove('hidden');
    contentElement.classList.add('hidden');
    
    fetch(`${restUrl}/actionscheduler-status`, {
        method: 'GET',
        headers: {
            'Content-Type': 'application/json',
            'X-WP-Nonce': wpApiSettings.nonce
        }
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.json();
    })
    .then(data => {
        // Aktualizace UI s daty
        document.getElementById('as-actions-size').textContent = data.actions_size || '0';
        document.getElementById('as-logs-size').textContent = data.logs_size || '0';
        document.getElementById('as-logs-count').textContent = data.logs_count || '0';
        
        // Aktualizace počtů akcí podle stavu
        if (data.action_counts) {
            document.getElementById('as-complete-count').textContent = data.action_counts.complete || '0';
            document.getElementById('as-pending-count').textContent = data.action_counts.pending || '0';
            document.getElementById('as-canceled-count').textContent = data.action_counts.canceled || '0';
            document.getElementById('as-failed-count').textContent = data.action_counts.failed || '0';
        }
        
        // Zobrazení obsahu
        loadingElement.classList.add('hidden');
        contentElement.classList.remove('hidden');
        
        // Přidání event listeneru pro tlačítko čištění
        const runAsCleanupButton = document.getElementById('run-as-cleanup');
        if (runAsCleanupButton) {
            runAsCleanupButton.addEventListener('click', runActionSchedulerCleanup);
        }
    })
    .catch(error => {
        console.error('Chyba při načítání stavu ActionScheduler:', error);
        loadingElement.innerHTML = '<div class="bg-red-100 text-red-700 p-4 rounded">Došlo k chybě při načítání stavu ActionScheduler.</div>';
    });
}

/**
 * Spuštění čištění ActionScheduler
 */
function runActionSchedulerCleanup() {
    setLoading(true);
    
    fetch(`${restUrl}/clean-actionscheduler`, {
        method: 'POST',
        headers: {
            'Content-Type': 'application/json',
            'X-WP-Nonce': wpApiSettings.nonce
        }
    })
    .then(response => {
        if (!response.ok) {
            throw new Error('Network response was not ok');
        }
        return response.json();
    })
    .then(data => {
        setLoading(false);
        if (data.success) {
            showNotification('Action Scheduler byl úspěšně vyčištěn.', 'success');
            
            // Znovu načtení dat
            loadActionSchedulerStatus();
        } else {
            showNotification('Při čištění Action Scheduler došlo k chybě.', 'error');
        }
    })
    .catch(error => {
        setLoading(false);
        console.error('Chyba při čištění ActionScheduler:', error);
        showNotification('Při čištění Action Scheduler došlo k chybě.', 'error');
    });
}
});
