/**
 * ZiziCache AI Admin Interface JavaScript
 * 
 * Handles the frontend interactions for the AI Optimizer admin panel
 */
(function($) {
    'use strict';
    
    const ZiziAI = {
        
        init: function() {
            this.bindEvents();
            this.updateStatus();
            this.initConfidenceSlider();
        },        bindEvents: function() {
            $('#start-analysis').on('click', this.startAnalysis.bind(this));
            $('#apply-recommendations').on('click', this.applyRecommendations.bind(this));
            $('#save-recommendations').on('click', this.saveRecommendations.bind(this));
            $('#ai-settings-form').on('submit', this.saveSettings.bind(this));
            $('#ai-config-form').on('submit', this.saveConfig.bind(this));
            $('#test-api-key').on('click', this.testAPI.bind(this));
            $('#refresh-models').on('click', this.refreshModels.bind(this));
            $('.view-analysis').on('click', this.viewAnalysis.bind(this));
            $('.apply-analysis').on('click', this.applyStoredAnalysis.bind(this));
            $('#confidence-threshold').on('input', this.updateConfidenceValue.bind(this));
        },
        
        initConfidenceSlider: function() {
            this.updateConfidenceValue();
        },
        
        updateConfidenceValue: function() {
            const slider = $('#confidence-threshold');
            const value = Math.round(slider.val() * 100);
            $('#confidence-value').text(value + '%');
        },
        
        startAnalysis: function(e) {
            e.preventDefault();
            
            const button = $(e.target);
            const spinner = button.siblings('.spinner');
            const url = $('#analysis-url').val();
            const type = $('#analysis-type').val();
            
            if (!url) {
                this.showNotice('Please enter a valid URL', 'error');
                return;
            }
            
            button.prop('disabled', true).text('🔄 ' + ziziAI.strings.analyzing);
            spinner.addClass('is-active');
            
            $.ajax({
                url: ziziAI.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'zizi_ai_analyze',
                    nonce: ziziAI.nonce,
                    url: url,
                    type: type
                },
                success: (response) => {
                    // response.success is true if wp_send_json_success was called from PHP,
                    // false if wp_send_json_error was called (or a jQuery AJAX transport error occurred).
                    if (response.success) {
                        const analysisPayload = response.data.analysis; // This is the $result from Connector::analyzeAndOptimize

                        if (analysisPayload && analysisPayload.success) {
                            // AI operation within Connector::analyzeAndOptimize was successful
                            if (analysisPayload.ai_recommendations) {
                                this.displayAnalysisResults(analysisPayload.ai_recommendations);
                                // Use the specific success message from the AJAX response if available, otherwise generic.
                                this.showNotice(response.data.message || ziziAI.strings.success, 'success');
                            } else {
                                // This is an unexpected state: AI operation reported success, but crucial data is missing.
                                this.showNotice('AI analysis result is incomplete. Missing recommendations data.', 'error');
                                console.error('Incomplete successful AI response: ai_recommendations missing.', analysisPayload);
                            }
                        } else {
                            // AI operation within Connector::analyzeAndOptimize failed (analysisPayload.success is false)
                            // or analysisPayload itself is missing/malformed.
                            let errorMessage = ziziAI.strings.error; // Default error message
                            if (analysisPayload && analysisPayload.error) {
                                errorMessage = analysisPayload.error; // Specific error message from the AI process
                            } else if (response.data && response.data.message && (!analysisPayload || typeof analysisPayload.success === 'undefined')) {
                                // Fallback if analysisPayload is weird but there's a general message from wp_send_json_success.
                                // This might occur if AdminInterface sends a success message but analysisPayload is unexpectedly empty.
                                errorMessage = response.data.message;
                            }
                            this.showNotice(errorMessage, 'error');
                            console.error('AI Analysis failed or returned an error:', {
                                analysisPayload: analysisPayload,
                                fullResponseData: response.data
                            });
                        }
                    } else {
                        // wp_send_json_error was called from PHP (e.g., nonce failure, permissions, or other Exception in AdminInterface)
                        // or a jQuery AJAX error occurred where it still invokes the success callback but with response.success = false.
                        let errorMessage = ziziAI.strings.error; // Default error
                        if (response.data && response.data.message) {
                            errorMessage = response.data.message; // Message from wp_send_json_error
                        }
                        this.showNotice(errorMessage, 'error');
                        console.error('AJAX request failed or returned error:', response.data);
                    }
                },
                error: () => {
                    this.showNotice(ziziAI.strings.error, 'error');
                },
                complete: () => {
                    button.prop('disabled', false).text('🚀 Start AI Analysis');
                    spinner.removeClass('is-active');
                }
            });
        },
          displayAnalysisResults: function(analysis) {
            const container = $('#analysis-results');
            const recommendationsContainer = container.find('.recommendations-container');
            
            let html = '<div class="analysis-summary">';
            html += '<div class="analysis-meta">';
            html += `<span class="confidence-badge" data-confidence="${analysis.metadata.confidence_score}">`;
            html += `Confidence: ${Math.round(analysis.metadata.confidence_score * 100)}%</span>`;
            html += `<span class="model-used">Model: ${analysis.metadata.ai_model}</span>`;
            html += `<span class="timestamp">${analysis.metadata.analysis_timestamp}</span>`;
            html += '</div>';
            
            if (analysis.validation && analysis.validation.warnings.length > 0) {
                html += '<div class="validation-warnings">';
                html += '<h4>⚠️ Validation Warnings:</h4>';
                html += '<ul>';
                analysis.validation.warnings.forEach(warning => {
                    html += `<li>${warning}</li>`;
                });
                html += '</ul>';
                html += '</div>';
            }
            
            html += '<div class="recommendations-list">';
            html += '<h4>✨ AI Recommendations:</h4>';
            
            if (analysis.recommendations && Object.keys(analysis.recommendations).length > 0) {
                html += this.createStructuredRecommendations(analysis.recommendations);
            } else {
                html += '<p class="no-recommendations">No optimization recommendations found.</p>';
            }
            
            html += '</div>';
            html += '</div>';
            
            recommendationsContainer.html(html);
            container.show();
            
            // Store recommendations for later use
            container.data('recommendations', analysis.recommendations);
        },
          createStructuredRecommendations: function(recommendations) {
            let html = '<div class="recommendations-sections">';
            
            // Handle structured recommendations with sections
            for (const [sectionKey, sectionData] of Object.entries(recommendations)) {
                if (typeof sectionData === 'object' && sectionData !== null && !Array.isArray(sectionData)) {
                    html += this.createRecommendationSection(sectionKey, sectionData);
                } else {
                    // Handle simple key-value pairs
                    html += this.createSimpleRecommendation(sectionKey, sectionData);
                }
            }
            
            html += '</div>';
            return html;
        },
        
        createRecommendationSection: function(sectionKey, sectionData) {
            const sectionTitle = this.humanizeSectionKey(sectionKey);
            const impact = this.getSectionImpact(sectionKey);
            
            let html = `<div class="recommendation-section section-${sectionKey} impact-${impact}">`;
            html += `<div class="section-header">`;
            html += `<h4>${this.getSectionIcon(sectionKey)} ${sectionTitle}</h4>`;
            html += `<span class="impact-badge">${impact.toUpperCase()}</span>`;
            html += `</div>`;
            html += `<div class="section-content">`;
            
            // Handle reasoning section specially
            if (sectionKey === 'reasoning') {
                html += this.createReasoningDisplay(sectionData);
            } else {
                html += '<div class="settings-grid">';
                for (const [key, value] of Object.entries(sectionData)) {
                    html += this.createSettingItem(key, value);
                }
                html += '</div>';
            }
            
            html += `</div>`;
            html += `</div>`;
            
            return html;
        },
        
        createReasoningDisplay: function(reasoning) {
            let html = '<div class="reasoning-content">';
            
            if (reasoning.website_type) {
                html += `<div class="reasoning-item">`;
                html += `<strong>🎯 Website Type:</strong> <span class="website-type">${reasoning.website_type}</span>`;
                html += `</div>`;
            }
            
            if (reasoning.optimization_focus) {
                html += `<div class="reasoning-item">`;
                html += `<strong>🔍 Focus:</strong> <span class="focus">${reasoning.optimization_focus}</span>`;
                html += `</div>`;
            }
            
            if (reasoning.key_recommendations && Array.isArray(reasoning.key_recommendations)) {
                html += `<div class="reasoning-item">`;
                html += `<strong>💡 Key Recommendations:</strong>`;
                html += `<ul class="key-recommendations">`;
                reasoning.key_recommendations.forEach(rec => {
                    html += `<li>${rec}</li>`;
                });
                html += `</ul>`;
                html += `</div>`;
            }
            
            html += '</div>';
            return html;
        },
        
        createSettingItem: function(key, value) {
            const humanKey = this.humanizeKey(key);
            const formattedValue = this.formatValue(value);
            const isRecommended = this.isRecommendedSetting(value);
            
            return `
                <div class="setting-item ${isRecommended ? 'recommended' : 'not-recommended'}">
                    <div class="setting-label">${humanKey}</div>
                    <div class="setting-value">${formattedValue}</div>
                </div>
            `;
        },
        
        createSimpleRecommendation: function(key, value) {
            return this.createRecommendationCard(key, value);
        },
        
        createRecommendationCard: function(key, value) {
            const humanKey = this.humanizeKey(key);
            const valueDisplay = this.formatValue(value);
            const impact = this.getImpactLevel(key);
            
            return `
                <div class="recommendation-card impact-${impact}">
                    <div class="recommendation-header">
                        <h5>${humanKey}</h5>
                        <span class="impact-badge">${impact}</span>
                    </div>
                    <div class="recommendation-value">
                        <strong>Recommended:</strong> ${valueDisplay}
                    </div>
                    <div class="recommendation-description">
                        ${this.getRecommendationDescription(key)}
                    </div>
                </div>
            `;
        },
        
        humanizeSectionKey: function(key) {
            const sectionMap = {
                'optimization': 'Core Optimization',
                'advanced': 'Advanced Features', 
                'cache': 'Cache Settings',
                'fonts': 'Font Optimization',
                'reasoning': 'AI Analysis Summary'
            };
            
            return sectionMap[key] || key.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
        },
        
        getSectionIcon: function(key) {
            const iconMap = {
                'optimization': '⚡',
                'advanced': '🔧',
                'cache': '💾',
                'fonts': '🔤',
                'reasoning': '🧠'
            };
            
            return iconMap[key] || '📋';
        },
        
        getSectionImpact: function(key) {
            const impactMap = {
                'optimization': 'high',
                'cache': 'high',
                'advanced': 'medium',
                'fonts': 'medium',
                'reasoning': 'info'
            };
            
            return impactMap[key] || 'low';
        },
          isRecommendedSetting: function(value) {
            if (typeof value === 'boolean') {
                return value; // true = recommended, false = not recommended
            }
            if (typeof value === 'number' && value > 0) {
                return true; // positive numbers generally recommended
            }
            if (Array.isArray(value) && value.length > 0) {
                return true; // non-empty arrays are recommendations
            }
            if (typeof value === 'string' && value.trim().length > 0) {
                return true; // non-empty strings are recommendations
            }
            return false;
        },
        
        humanizeKey: function(key) {
            const keyMap = {
                'enable_page_cache': 'Page Cache',
                'enable_object_cache': 'Object Cache',
                'minify_html': 'HTML Minification',
                'minify_css': 'CSS Minification',
                'minify_js': 'JavaScript Minification',
                'combine_css': 'CSS Combination',
                'combine_js': 'JavaScript Combination',
                'defer_js': 'JavaScript Deferring',
                'lazy_load_images': 'Image Lazy Loading',
                'webp_conversion': 'WebP Conversion',
                'preload_fonts': 'Font Preloading',
                'cache_expiry': 'Cache Expiry Time',
                'compression_level': 'Cache Compression'
            };
            
            return keyMap[key] || key.replace(/_/g, ' ').replace(/\b\w/g, l => l.toUpperCase());
        },
          formatValue: function(value) {
            if (typeof value === 'boolean') {
                return value ? '✅ Enable' : '❌ Disable';
            }
            if (typeof value === 'number') {
                // Format numbers with units where applicable
                if (value >= 86400) {
                    return Math.round(value / 86400) + ' days';
                } else if (value >= 3600) {
                    return Math.round(value / 3600) + ' hours';
                } else if (value >= 60) {
                    return Math.round(value / 60) + ' minutes';
                } else if (value >= 1000) {
                    return value + 'ms';
                } else {
                    return value.toString();
                }
            }
            if (Array.isArray(value)) {
                if (value.length === 0) {
                    return '<em>None</em>';
                }
                return '<ul class="value-list">' + value.map(item => `<li>${this.formatValue(item)}</li>`).join('') + '</ul>';
            }
            if (typeof value === 'object' && value !== null) {
                // Handle nested objects
                let result = '<div class="nested-object">';
                for (const [key, val] of Object.entries(value)) {
                    result += `<div class="object-item"><strong>${this.humanizeKey(key)}:</strong> ${this.formatValue(val)}</div>`;
                }
                result += '</div>';
                return result;
            }
            if (typeof value === 'string') {
                // Handle URLs
                if (value.startsWith('http://') || value.startsWith('https://')) {
                    const shortUrl = value.length > 50 ? value.substring(0, 47) + '...' : value;
                    return `<a href="${value}" target="_blank" class="url-link">${shortUrl}</a>`;
                }
                return value;
            }
            return String(value);
        },
        
        getImpactLevel: function(key) {
            const highImpact = ['enable_page_cache', 'minify_css', 'minify_js', 'lazy_load_images'];
            const mediumImpact = ['combine_css', 'combine_js', 'webp_conversion', 'defer_js'];
            
            if (highImpact.includes(key)) return 'high';
            if (mediumImpact.includes(key)) return 'medium';
            return 'low';
        },
        
        getRecommendationDescription: function(key) {
            const descriptions = {
                'enable_page_cache': 'Enables full-page caching to dramatically reduce server load and improve response times.',
                'minify_css': 'Removes whitespace and comments from CSS files to reduce file size.',
                'minify_js': 'Removes whitespace and comments from JavaScript files to reduce file size.',
                'lazy_load_images': 'Loads images only when they become visible to improve initial page load speed.',
                'webp_conversion': 'Converts images to WebP format for better compression and faster loading.',
                'defer_js': 'Delays JavaScript execution until after page content is loaded.',
                'combine_css': 'Combines multiple CSS files into one to reduce HTTP requests.',
                'combine_js': 'Combines multiple JavaScript files into one to reduce HTTP requests.'
            };
            
            return descriptions[key] || 'Optimization recommendation based on AI analysis.';
        },
        
        applyRecommendations: function(e) {
            e.preventDefault();
            
            const button = $(e.target);
            const recommendations = $('#analysis-results').data('recommendations');
            
            if (!recommendations) {
                this.showNotice('No recommendations to apply', 'error');
                return;
            }
            
            if (!confirm('Are you sure you want to apply these AI recommendations? This will modify your plugin settings.')) {
                return;
            }
            
            button.prop('disabled', true).text('⚙️ ' + ziziAI.strings.applying);
            
            $.ajax({
                url: ziziAI.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'zizi_ai_apply_recommendations',
                    nonce: ziziAI.nonce,
                    recommendations: JSON.stringify(recommendations)
                },
                success: (response) => {
                    if (response.success) {
                        this.showNotice(ziziAI.strings.applied, 'success');
                        this.updateStatus();
                        // Optionally reload the page to reflect changes
                        setTimeout(() => location.reload(), 2000);
                    } else {
                        this.showNotice(response.data.message || 'Failed to apply recommendations', 'error');
                    }
                },
                error: () => {
                    this.showNotice('Failed to apply recommendations', 'error');
                },
                complete: () => {
                    button.prop('disabled', false).text('✨ Apply Recommendations');
                }
            });
        },
        
        saveRecommendations: function(e) {
            e.preventDefault();
            
            const recommendations = $('#analysis-results').data('recommendations');
            
            if (!recommendations) {
                this.showNotice('No recommendations to save', 'error');
                return;
            }
            
            // Here you could implement saving to browser storage or server
            const analysisData = {
                recommendations: recommendations,
                timestamp: new Date().toISOString(),
                url: $('#analysis-url').val(),
                type: $('#analysis-type').val()
            };
            
            localStorage.setItem('zizi_ai_saved_analysis', JSON.stringify(analysisData));
            this.showNotice('Recommendations saved for later', 'success');
        },
        
        saveSettings: function(e) {
            e.preventDefault();
            
            const form = $(e.target);
            const button = form.find('button[type="submit"]');
            
            button.prop('disabled', true).text('Saving...');
            
            // Here you would save the settings via AJAX
            setTimeout(() => {
                this.showNotice('AI settings saved successfully', 'success');
                button.prop('disabled', false).text('Save AI Settings');
            }, 1000);
        },
        
        updateStatus: function() {
            $.ajax({
                url: ziziAI.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'zizi_ai_get_status',
                    nonce: ziziAI.nonce
                },
                success: (response) => {
                    if (response.success) {
                        this.updateStatusDisplay(response.data);
                    }
                }
            });
        },
        
        updateStatusDisplay: function(data) {
            // Update daily requests progress
            const progressBar = $('.progress-fill');
            const dailyRequests = data.usage.daily_requests || 0;
            const percentage = (dailyRequests / 50) * 100;
            progressBar.css('width', percentage + '%');
            
            // Update other status indicators as needed
        },
        
        viewAnalysis: function(e) {
            e.preventDefault();
            
            const analysisId = $(e.target).data('id');
            
            // Implement modal or expand functionality to show analysis details
            this.showNotice('Analysis details would be shown here', 'info');
        },
        
        applyStoredAnalysis: function(e) {
            e.preventDefault();
            
            const analysisId = $(e.target).data('id');
            
            if (!confirm('Apply this stored analysis?')) {
                return;
            }
            
            // Implement applying stored analysis
            this.showNotice('Stored analysis would be applied here', 'info');
        },
        
        showNotice: function(message, type = 'info') {
            const noticeClass = type === 'error' ? 'notice-error' : 
                               type === 'success' ? 'notice-success' : 
                               'notice-info';
            
            const notice = $(`
                <div class="notice ${noticeClass} is-dismissible">
                    <p>${message}</p>
                    <button type="button" class="notice-dismiss">
                        <span class="screen-reader-text">Dismiss this notice.</span>
                    </button>
                </div>
            `);
            
            $('.wrap h1').after(notice);
            
            // Auto dismiss after 5 seconds
            setTimeout(() => {
                notice.fadeOut(() => notice.remove());
            }, 5000);
              // Manual dismiss
            notice.find('.notice-dismiss').on('click', function() {
                notice.fadeOut(() => notice.remove());
            });
        },
        
        saveConfig: function(e) {
            e.preventDefault();
            
            const form = $(e.target);
            const button = form.find('button[type="submit"]');
            const apiKey = $('#openrouter-api-key').val().trim();
            const model = $('#ai-model').val();
            
            if (!apiKey) {
                this.showNotice('Please enter an API key', 'error');
                return;
            }
            
            button.prop('disabled', true).text('💾 Saving...');
            
            $.ajax({
                url: ziziAI.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'zizi_ai_save_config',
                    nonce: $('[name="zizi_ai_config_nonce"]').val(),
                    api_key: apiKey,
                    model: model
                },
                success: (response) => {
                    if (response.success) {
                        this.showNotice(response.data.message, 'success');
                        this.updateStatus();
                    } else {
                        this.showNotice(response.data.message || 'Failed to save configuration', 'error');
                    }
                },
                error: () => {
                    this.showNotice('Failed to save configuration', 'error');
                },
                complete: () => {
                    button.prop('disabled', false).text('💾 Save Configuration');
                }
            });
        },
        
        testAPI: function(e) {
            e.preventDefault();
            
            const button = $(e.target);
            const apiKey = $('#openrouter-api-key').val().trim();
            
            if (!apiKey) {
                this.showNotice('Please enter an API key first', 'error');
                return;
            }
            
            button.prop('disabled', true).text('🧪 Testing...');
            
            $.ajax({
                url: ziziAI.ajaxUrl,
                type: 'POST',                data: {
                    action: 'zizi_ai_test_api',
                    nonce: ziziAI.nonce
                },
                success: (response) => {
                    if (response.success) {
                        this.showNotice(response.data.message, 'success');
                    } else {
                        this.showNotice(response.data || 'API test failed', 'error');
                    }
                },
                error: () => {
                    this.showNotice('API test failed', 'error');
                },                complete: () => {
                    button.prop('disabled', false).text('🧪 Test API Connection');
                }
            });
        },
        
        refreshModels: function(e) {
            e.preventDefault();
            
            const button = $(e.target);
            const spinner = button.siblings('.spinner');
            const select = $('#ai-model');
            
            button.prop('disabled', true).text('🔄 Refreshing...');
            spinner.addClass('is-active');
            
            $.ajax({
                url: ziziAI.ajaxUrl,
                type: 'POST',
                data: {
                    action: 'zizi_ai_refresh_models',
                    nonce: ziziAI.nonce
                },
                success: (response) => {
                    if (response.success) {
                        // Update the select options
                        select.html(response.data.options_html);
                        this.showNotice(response.data.message, 'success');
                    } else {
                        this.showNotice(response.data.message || 'Failed to refresh models', 'error');
                    }
                },
                error: () => {
                    this.showNotice('Failed to refresh models', 'error');
                },
                complete: () => {
                    button.prop('disabled', false).text('🔄 Refresh Models');
                    spinner.removeClass('is-active');
                }
            });
        }
    };
    
    // Initialize when document is ready
    $(document).ready(function() {
        ZiziAI.init();
    });
    
})(jQuery);

// Add CSS for the recommendations display
const aiAdminCSS = `
<style>
.analysis-summary {
    border: 1px solid #ddd;
    border-radius: 8px;
    padding: 20px;
    background: #fff;
}

.analysis-meta {
    display: flex;
    gap: 15px;
    margin-bottom: 20px;
    flex-wrap: wrap;
}

.confidence-badge {
    padding: 4px 12px;
    border-radius: 20px;
    font-size: 12px;
    font-weight: bold;
}

.confidence-badge[data-confidence^="0.9"], 
.confidence-badge[data-confidence="1"] {
    background: #d4edda;
    color: #155724;
}

.confidence-badge[data-confidence^="0.8"] {
    background: #fff3cd;
    color: #856404;
}

.confidence-badge[data-confidence^="0.7"],
.confidence-badge[data-confidence^="0.6"] {
    background: #f8d7da;
    color: #721c24;
}

.model-used, .timestamp {
    padding: 4px 8px;
    background: #f1f1f1;
    border-radius: 4px;
    font-size: 11px;
    color: #666;
}

.validation-warnings {
    background: #fff3cd;
    border: 1px solid #ffeaa7;
    border-radius: 4px;
    padding: 15px;
    margin-bottom: 20px;
}

.validation-warnings h4 {
    margin-top: 0;
    color: #856404;
}

.recommendations-grid {
    display: grid;
    grid-template-columns: repeat(auto-fill, minmax(300px, 1fr));
    gap: 15px;
    margin-top: 15px;
}

.recommendation-card {
    border: 1px solid #ddd;
    border-radius: 6px;
    padding: 15px;
    background: #fafafa;
}

.recommendation-card.impact-high {
    border-left: 4px solid #dc3545;
}

.recommendation-card.impact-medium {
    border-left: 4px solid #ffc107;
}

.recommendation-card.impact-low {
    border-left: 4px solid #28a745;
}

.recommendation-header {
    display: flex;
    justify-content: space-between;
    align-items: center;
    margin-bottom: 10px;
}

.recommendation-header h5 {
    margin: 0;
    font-size: 14px;
}

.impact-badge {
    font-size: 10px;
    padding: 2px 6px;
    border-radius: 10px;
    text-transform: uppercase;
    font-weight: bold;
}

.impact-badge.high {
    background: #dc3545;
    color: white;
}

.impact-badge.medium {
    background: #ffc107;
    color: #212529;
}

.impact-badge.low {
    background: #28a745;
    color: white;
}

.recommendation-value {
    margin-bottom: 8px;
    font-size: 13px;
}

.recommendation-description {
    font-size: 12px;
    color: #666;
    line-height: 1.4;
}

.no-recommendations {
    text-align: center;
    color: #666;
    font-style: italic;    padding: 30px;
}

/* Configuration Panel Styles */
.zizi-ai-config-panel {
    background: #fff;
    border: 1px solid #ddd;
    border-radius: 8px;
    padding: 20px;
    margin-bottom: 20px;
}

.zizi-ai-config-panel h2 {
    margin-top: 0;
    color: #333;
    display: flex;
    align-items: center;
    gap: 8px;
}

.config-form .form-group {
    margin-bottom: 20px;
}

.config-form label {
    display: block;
    font-weight: 600;
    margin-bottom: 5px;
    color: #333;
}

.config-form input[type="password"],
.config-form select {
    width: 100%;
    max-width: 400px;
}

.config-form .description {
    margin-top: 5px;
    font-style: italic;
    color: #666;
}

.config-form .form-actions {
    display: flex;
    gap: 10px;
    align-items: center;
}

.form-actions .button {
    display: inline-flex;
    align-items: center;
    gap: 5px;
}

@media (max-width: 768px) {
    .recommendations-grid {
        grid-template-columns: 1fr;
    }
    
    .analysis-meta {
        flex-direction: column;
    }
}
</style>
`;

// Inject CSS into the page
$(document).ready(function() {
    $('head').append(aiAdminCSS);
});
